#!/usr/bin/env python3

# Brevdue i Europa (Norwegian) licence patcher
#
# Licence information is compiled into the executable and obfuscated with a XOR scheme.
# This script lets you change this information displayed in the credits screen.
#
# ./patch_licence.py /path/to/RUNKO.EXE "Your Name"
#
# This will write the modified executable to /path/to/RUNKO.EXE.patched.
#
# The distributor field can also be updated with the optional --dsitributor argument.
# Default value is "Statens Filmsentral.".

import argparse
import hashlib
import os
import sys

FILE_SIZE = 119440
FILE_SHA1 = '08228da8a44478b6b4eaeb4e253e5a7a3c34dacc'

LICENCEE1_OFF = 16342
LICENCEE1_LEN = 60
LICENCEE2_OFF = 16444
LICENCEE2_LEN = 61
LICENCEE_KEY_OFF = 118704

DISTRIBUTOR_OFF = 16403
DISTRIBUTOR_LEN = 40
DISTRIBUTOR_KEY_OFF = 118960

parser = argparse.ArgumentParser(description='Brevdue i Europa (Norwegian) licence patcher')
parser.add_argument('file_path', type=str, help='Path to RUNKO.EXE')
parser.add_argument('licencee', type=str, help='Name of licencee')
parser.add_argument('--distributor', type=str, default='Statens Filmsentral.', help='Name of distributor (default: "Statens Filmsentral.")')
args = parser.parse_args()

def clear_offset(file_data, offset, length):
    file_data[offset:offset + length] = b'\x00' * length
    return file_data

def patch_offset(file_data, offset, data):
    file_data[offset:offset + len(data)] = data
    return file_data

def read_executable(file_path):
    # Verify file size
    file_size = os.path.getsize(file_path)
    if file_size != FILE_SIZE:
        raise ValueError(f'File size for "{file_path}" is {file_size} bytes, expected {FILE_SIZE} bytes.')

    # Read entire file into memory
    with open(file_path, 'rb') as f:
        file_data = bytearray(f.read())

    # Clear existing licence values
    file_data = clear_offset(file_data, LICENCEE1_OFF, LICENCEE1_LEN)
    file_data = clear_offset(file_data, LICENCEE2_OFF, LICENCEE2_LEN)
    file_data = clear_offset(file_data, DISTRIBUTOR_OFF, DISTRIBUTOR_LEN)

    # Check if remaining data matches hash
    sha1 = hashlib.sha1(file_data).hexdigest()
    if sha1 != FILE_SHA1:
        print(f'SHA1 for "{file_path}" with licence fields cleared is {sha1}, expected {FILE_SHA1}. Patching may fail.', file=sys.stderr)

    return file_data    

# Extract an existing key from the executable. It is not known whether this key is tied to the licence.
def extract_key(file_data, offset):
    length = file_data[offset]
    # The game code has an off-by-one error when repeating the key. The last character of the key is substituted with the key's preceeding length counter.
    key = file_data[offset + 1:offset + length]
    key.append(length)
    return key

def encode(name, key, length, xoronly = False):
    # Ensure name is exactly the given length and encoded as cp865
    name = bytes((name[:length] + ' ' * length)[:length].encode('cp865'))
    # Make key the same length as the name.
    key = (key * (len(name) // len(key) + 1))[:len(name)]

    if not xoronly:
        # Pass 1: Add 0x0B to every character
        name = bytes((c + 0x0B) & 0xFF for c in name)

        # Pass 2: Subtract key from name
        name = bytes((c - k) & 0xFF for c, k in zip(name, key))

        # Pass 3: Add character position to every character
        name = bytes((c + i + 1) & 0xFF for i, c in enumerate(name))

    # Last pass: XOR name with key
    return bytes(c ^ k for c, k in zip(name, key))

def decode(name, key, xoronly = False):
    # Make key the same length as the name.
    key = (key * (len(name) // len(key) + 1))[:len(name)]

    # Pass 1, XOR name with key
    name = bytes(c ^ k for c, k in zip(name, key))

    if (not xoronly):
        # Pass 2: Subtract character position from every character
        name = bytes((c - i - 1) & 0xFF for i, c in enumerate(name))

        # Pass 3: Add key to name
        name = bytes((c + k) & 0xFF for c, k in zip(name, key))

        # Pass 4: Subtract 0x0B from every character
        name = bytes((c - 0x0B) & 0xFF for c in name)

    return name

file_data = read_executable(args.file_path)
licencee_key = extract_key(file_data, LICENCEE_KEY_OFF)
distributor_key = extract_key(file_data, DISTRIBUTOR_KEY_OFF)

print(f'licencee_key: {licencee_key.decode()}')
print(f'distributor_key: {distributor_key.decode()}')

encoded_licencee1 = encode(args.licencee, licencee_key, LICENCEE1_LEN)
encoded_licencee2 = encode(args.licencee, licencee_key, LICENCEE2_LEN)
encoded_distributor = encode(args.distributor, distributor_key, DISTRIBUTOR_LEN, xoronly = True)

print(f'licencee: {args.licencee}')
print(f'encoded_licencee1: {encoded_licencee1}')
print(f'encoded_licencee1: {encoded_licencee2}')
print(f'distributor: {args.distributor}')
print(f'encoded_distributor: {encoded_distributor}')

file_data = patch_offset(file_data, LICENCEE1_OFF, encoded_licencee1)
file_data = patch_offset(file_data, LICENCEE2_OFF, encoded_licencee2)
file_data = patch_offset(file_data, DISTRIBUTOR_OFF, encoded_distributor)

patched_file_path = args.file_path + '.patched'
print(f'Writing "{patched_file_path}"...')

with open(patched_file_path, 'wb') as f:
    f.write(file_data)

print('Done.')
