{ HANOWE.PAS - Towers of Hanoi

  Title   : HANOWE
  Language: Borland Pascal v7.0 with Object Windows
  Version : 1.5
  Date    : Feb 9, 2000
  Author  : J R Ferguson
  Usage   : MS-Windows 3.1 application
  Download: http://hello.to/ferguson
  E-mail  : j.r.ferguson@iname.com

This program and its source may be used and copied freely without charge,
but  only  for non-commercial purposes. The author is not responsible for
any damage or loss of data that may be caused by using it.
}

{$B-} { short-circuit Boolean expression evaluation }

program HANOWE;

uses WinTypes, WinProcs, OWindows, Objects, ODialogs, OStdDlgs;

{ --- Resources --- }

{$R HANOWE.RES}
const
  cm_Options   = 101;
  cm_Begin     = 102;
  cm_Auto      = 103;
  cm_GameExit  = 104;
  cm_Rules     = 111;
  cm_About     = 112;

  cm_NumBase   = 200;
  id_Speed     = 211;


{ -- Global declarations --- }

  C_MaxDisk    = 10;    { max number of disks }
  C_DflDisk    = 5;     { default number of disks }
  C_MaxTower   = 3;     { number of towers }
  C_TimeBase   = 1000;  { fixed start value for wait counter }
  C_TimeFactor = 8000;  { decides speed for wait counter }
  C_MinSpeed   = 0;     { minimum speed }
  C_DflSpeed   = 5;     { default speed }
  C_MaxSpeed   = 10;    { maximum speed }
  C_DiskXUnit  = 10;
  C_DiskYUnit  = 16;
  C_TowerXUnit = 2 * (C_MaxDisk + 1) * C_DiskXUnit;
  C_TowerYUnit = C_DiskYUnit div 2;
  C_XOrigin    = 12 * C_DiskXUnit;
  C_YOrigin    = (C_MaxDisk + 5)  * C_DiskYUnit;
  C_MaxSeqCnt  = (2 shl C_MaxDisk) - 1;
  C_MaxSeqBuf  = (2 shl (C_MaxDisk - 1));

type
  T_DiskInd    = 0..C_MaxDisk;
  T_TowerInd   = 0..C_MaxTower;
  T_SeqCount   = 0..C_MaxSeqCnt;
  T_SeqBufInd  = 0..C_MaxSeqBuf;
  T_OptionBuf  = record
                   Count : array[1..C_MaxDisk] of word;
                   Speed : record LowVal, HighVal, Position: integer; end;
                 end;

  P_Disk       = ^T_Disk;
  P_Tower      = ^T_Tower;
  P_Sequence   = ^T_Sequence;
  P_OptionDlg  = ^T_OptionDlg;
  P_OptionBuf  = ^T_OptionBuf;
  P_GameWindow = ^T_GameWindow;

  T_Disk = object(TObject)
    X,Y    : integer;
    Index  : T_DiskInd;
    Tower  : P_Tower;
    Next   : P_Disk;
    constructor Init(V_Tower: P_Tower; V_Index: T_DiskInd);
    destructor  Done; virtual;
    procedure   Draw(V_DC: HDC; V_Pen: HPen);
    procedure   Move(V_NewX, V_NewY: integer);
    function    Hit(V_X, V_Y: integer): boolean;
  end;

  T_Tower = object(TObject)
    X      : integer;
    Top    : P_Disk;
    constructor Init(V_Index: T_TowerInd);
    destructor  Done; virtual;
    procedure   DrawTower(V_DC: HDC; V_Pen: HPen);
    procedure   DrawDisks(V_DC: HDC; V_Pen: HPen);
    function    Connect(V_Disk: P_Disk): boolean;
    function    GetTop: P_Disk;
    procedure   RemoveDisks;
    function    SrcHit(V_X, V_Y: integer): boolean;
    function    DstHit(V_X, V_Y: integer): boolean;
  end;

  T_Sequence = object(TObject)
    Buffer      : array[1..C_MaxSeqBuf] of byte;
    Index       : T_SeqBufInd;
    Count       : T_SeqCount;
    MaxCount    : T_SeqCount;
    constructor   Init;
    destructor    Done; virtual;
    procedure     Fill(V_DiskCount: T_DiskInd);
    function      GetMove(var V_Src, V_Dst: T_TowerInd): boolean;
  end;

  T_OptionDlg = object(TDialog)
    constructor Init(V_Parent: PWindowsObject; V_Name: PChar; V_OptionBuf: P_OptionBuf);
  end;

  T_GameWindow = object(TWindow)
    Context     : HDC;
    DiskShowPen : HPen;
    DiskHidePen : HPen;
    TowerShowPen: HPen;
    TowerHidePen: HPen;
    Tower       : array[1..C_MaxTower] of P_Tower;
    DiskCount   : T_DiskInd;
    Auto        : boolean;
    AutoSpeed   : integer;
    DragDisk    : P_Disk;
    IdleCount   : longint;
    Sequence    : P_Sequence;
    OptBuf      : T_OptionBuf;
  { general }
    constructor Init(V_Parent: PWindowsObject; V_Title: PChar);
    destructor  Done; virtual;
    function    GetClassName: PChar; virtual;
    procedure   GetWindowClass(var V_Class: TWndClass); virtual;
    procedure   Draw(V_DC: HDC);
    procedure   Paint(V_DC: HDC; var V_PaintInfo: TPaintStruct); virtual;
    procedure   GameInit;
    procedure   GameReset;
    procedure   GameDone;
    procedure   MoveDragDisk(V_X, V_Y: integer);
  { command messages }
    procedure   CMOptions    (var V_Msg: TMessage); virtual cm_First + cm_Options;
    procedure   CMBegin      (var V_Msg: TMessage); virtual cm_First + cm_Begin;
    procedure   CMAuto       (var V_Msg: TMessage); virtual cm_First + cm_Auto;
    procedure   CMGameExit   (var V_Msg: TMessage); virtual cm_First + cm_GameExit;
    procedure   CMRules      (var V_Msg: TMessage); virtual cm_First + cm_Rules;
    procedure   CMAbout      (var V_Msg: TMessage); virtual cm_First + cm_About;
  { window messages }
    procedure   WMLButtonDown(var V_Msg: TMessage); virtual wm_First + wm_LButtonDown;
    procedure   WMLButtonUp  (var V_Msg: TMessage); virtual wm_First + wm_LButtonUp;
    procedure   WMMouseMove  (var V_Msg: TMessage); virtual wm_First + wm_MouseMove;
  { auto mode handling }
    function    IdleAction: boolean;
    function    AutoMove: boolean;
    procedure   AutoStart;
  end;

  T_Application = object(TApplication)
    procedure   InitMainWindow; virtual;
    procedure   InitInstance; virtual;
    function    IdleAction: boolean; virtual;
  end;

var
  G_Application : T_Application;

{ --- General routines --- }

function SpeedCount(V_Speed: integer): longint;
begin SpeedCount:= C_TimeBase + longint(C_MaxSpeed - V_Speed) * C_TimeFactor; end;

{ --- T_Disk methods --- }

constructor T_Disk.Init(V_Tower: P_Tower; V_Index: T_DiskInd);
begin
  Index:= V_Index;
  if not V_Tower^.Connect(@Self) then begin
    X:= 0; Y:= 0;
    Tower:= nil;
    Next := nil;
  end;
end;

destructor T_Disk.Done;
begin inherited Done; end;

procedure T_Disk.Draw(V_DC: HDC; V_Pen: HPen);
var radius: integer;
begin
  SelectObject(V_DC, V_Pen);
  radius:= Index * C_DiskXUnit;
  MoveTo(V_DC, X - radius, Y);
  LineTo(V_DC, X + radius, Y);
end;

procedure T_Disk.Move(V_NewX, V_NewY: integer);
begin X:= V_NewX; Y:= V_NewY; end;

function T_Disk.Hit(V_X, V_Y: integer): boolean;
const Yradius = C_DiskYUnit div 2;
var   Xradius: integer;
begin
  Xradius:= C_MaxDisk * C_DiskXUnit;
  Hit:= (V_X >= X - Xradius ) and (V_X <= X + Xradius) and (V_Y <= Y + Yradius);
end;

{--- T_Tower methods --- }

constructor T_Tower.Init(V_Index: T_TowerInd);
begin
  inherited Init;
  X   := C_XOrigin + (V_Index - 1) * C_TowerXUnit;
  Top := nil;
end;

destructor T_Tower.Done;
begin
  RemoveDisks;
  inherited Done;
end;

procedure T_Tower.DrawTower(V_DC: HDC; V_Pen: HPen);
var radius: integer;
begin
  SelectObject(V_DC, V_Pen);
  radius:= C_MaxDisk * C_DiskXUnit;
  MoveTo(V_DC, X - radius, C_YOrigin + C_DiskYUnit);
  LineTo(V_DC, X + radius, C_YOrigin + C_DiskYUnit);
end;

procedure T_Tower.DrawDisks(V_DC: HDC; V_Pen: HPen);
var p: P_Disk;
begin
  p:= Top;
  while p <> nil do begin
    p^.Draw(V_DC, V_Pen);
    p:= p^.Next;
  end;
end;

function T_Tower.Connect(V_Disk: P_Disk): boolean;
begin
  if (Top <> nil) and (Top^.Index < V_Disk^.Index) then Connect:= false
  else begin
    with V_Disk^ do begin
      X    := Self.X;
      if Top = nil then Y:= C_YOrigin else Y := Top^.Y - C_DiskYUnit;
      Tower:= @Self;
      Next := Top;
    end;
    Top:= V_Disk;
    Connect:= true;
  end;
end;

function T_Tower.GetTop: P_Disk;
var p: P_Disk;
begin
  p:= Top;
  if p <> nil then begin
    Top:= p^.Next;
    p^.Next := nil;
  end;
  GetTop:= p;
end;

procedure T_Tower.RemoveDisks;
var p: P_Disk;
begin
  p:= GetTop;
  while p <> nil do begin
    Dispose(p,Done);
    p:= GetTop;
  end;
end;

function T_Tower.SrcHit(V_X, V_Y: integer): boolean;
begin
  SrcHit:= (Top <> nil) and Top^.Hit(V_X, V_Y)
end;

function T_Tower.DstHit(V_X, V_Y: integer): boolean;
var Xradius: integer;
begin
  Xradius:=  C_MaxDisk * C_DiskXUnit;
  DstHit:= (V_X >= X - Xradius ) and (V_X <= X + Xradius)
end;


{ --- T_Sequence methods --- }

constructor T_Sequence.Init;
begin
  inherited Init;
  Index:= 0; Count:= 0; MaxCount:= 0;
end;

destructor T_Sequence.Done;
begin
  Index:= 0; Count:= 0; MaxCount:= 0;
  inherited Done;
end;

procedure T_Sequence.Fill(V_DiskCount: T_DiskInd);
  procedure InsertMove(V_Src, V_Dst: T_TowerInd);
    var nibble: byte;
    begin
      nibble:= (V_Src shl 2) + V_dst; Inc(MaxCount);
      if odd(MaxCount) then begin Inc(Index); Buffer[Index]:= Nibble; end
      else Buffer[Index]:= Buffer[Index] or (nibble shl 4);
    end;
  procedure ScanMoves(V_DiskCount: T_DiskInd; V_Src, V_Tmp, V_Dst: T_TowerInd);
    begin if V_DiskCount > 0 then begin
      ScanMoves(V_DiskCount - 1, V_Src, V_Dst, V_Tmp);
      InsertMove(V_Src, V_Dst);
      ScanMoves(V_DiskCount - 1, V_Tmp, V_Src, V_Dst);
    end end;
begin {Fill}
  Index:= 0; Count:= 0; MaxCount:= 0;
  ScanMoves(V_DiskCount, 1, 2, 3);
  Index:= 0;
end;

function T_Sequence.GetMove(var V_Src, V_Dst: T_TowerInd): boolean;
var nibble: byte;
begin
  if Count < MaxCount then begin
    Inc(Count);
    if odd(Count) then begin Inc(Index); nibble:= Buffer[Index] and $0F; end
    else nibble := Buffer[Index] shr 4;
    V_Src:= nibble shr 2; V_Dst:= nibble and $03;
    GetMove:= true;
  end
  else GetMove:= false;
end;

{ --- T_OptionDlg methods --- }

constructor T_OptionDlg.Init(V_Parent: PWindowsObject; V_Name: PChar; V_OptionBuf: P_OptionBuf);
var p: PControl; i: T_DiskInd;
begin
  inherited Init(V_Parent, V_Name);
  for i:= 1 to C_MaxDisk do p:= New(PRadioButton,InitResource(@Self, cm_NumBase + i));
  p:= New(PScrollBar,InitResource(@Self, id_Speed));
  TransferBuffer:= V_OptionBuf;
end;

{ --- T_GameWindow methods --- }

constructor T_GameWindow.Init(V_Parent: PWindowsObject; V_Title: PChar);
begin
  inherited Init(V_Parent, V_Title);
  Attr.Menu    := LoadMenu(HInstance,'MENU');
  Attr.W       := 2 * (C_XOrigin + C_TowerXUnit);
  Attr.H       := 64 + C_YOrigin + 2 * C_DiskYUnit;
  DiskCount    := C_DflDisk;
  Auto         := false;
  AutoSpeed    := C_DflSpeed;
  DragDisk     := nil;
  DiskShowPen  := CreatePen(PS_SOLID,C_DiskYUnit-1,RGB(000,127,127));
  DiskHidePen  := CreatePen(PS_SOLID,C_DiskYUnit-1,RGB(255,255,255));
  TowerShowPen := CreatePen(PS_SOLID,C_TowerYUnit ,RGB(127,127,127));
  TowerHidePen := CreatePen(PS_SOLID,C_TowerYUnit ,RGB(255,255,255));
  New(Sequence, Init);
  GameInit;
end;

destructor T_GameWindow.Done;
begin
  GameDone;
  Dispose(Sequence,Done);
  DeleteObject(DiskShowPen);
  DeleteObject(DiskHidePen);
  DeleteObject(TowerShowPen);
  DeleteObject(TowerHidePen);
  inherited Done;
end;

procedure T_GameWindow.Draw(V_DC: HDC);
var t: T_TowerInd; OldBrush: HBrush;
begin
  FloodFill(V_DC, 1,1, RGB(255,255,255));
  for t:= 1 to C_MaxTower do with Tower[t]^ do begin
    DrawTower(V_DC, TowerShowPen);
    DrawDisks(V_DC, DiskShowPen);
  end;
end;

function T_GameWindow.GetClassName: PChar;
begin GetClassName:= 'GameWindow'; end;

procedure T_GameWindow.GetWindowClass(var V_Class: TWndClass);
begin
  inherited GetWindowClass(V_Class);
  V_Class.hIcon := LoadIcon(HInstance,'ICON');
end;

procedure T_GameWindow.Paint(V_DC: HDC; var V_PaintInfo: TPaintStruct);
begin Draw(V_DC) end;

procedure T_GameWindow.GameInit;
var
  t : T_TowerInd;
  d : T_DiskInd;
  p : P_Disk;
begin
  for t:= 1 to C_MaxTower do New(Tower[t],Init(t));
  for d:= DiskCount downto 1 do New(p,Init(Tower[1], d));
end;

procedure T_GameWindow.GameReset;
var
  t : T_TowerInd;
  d : T_DiskInd;
  p : P_Disk;
begin
  Context:= GetDC(HWindow);
  for t:= 1 to C_MaxTower do with Tower[t]^ do begin
    DrawDisks(Context, DiskHidePen);
    RemoveDisks;
  end;
  for d:= DiskCount downto 1 do New(p,Init(Tower[1], d));
  DragDisk:= nil;
  Draw(Context);
  ReleaseDC(HWindow,Context);
end;

procedure T_GameWindow.GameDone;
var t: T_TowerInd;
begin
  DiskCount:= 0; GameReset;
  Context:= GetDC(HWindow);
  for t:= 1 to C_MaxTower do begin
    Tower[t]^.DrawTower(Context, TowerHidePen);
    Dispose(Tower[t],Done);
  end;
  ReleaseDC(HWindow,Context);
end;

procedure T_GameWindow.CMOptions(var V_Msg: TMessage);
var i: T_DiskInd; Count: T_DiskInd;
begin
  Count:= DiskCount;
  for i:= 1 to C_MaxDisk do if i= Count then OptBuf.Count[i]:= bf_Checked
                                        else OptBuf.Count[i]:= bf_Unchecked;
  with OptBuf.Speed do begin
    LowVal:= C_MinSpeed; HighVal:= C_MaxSpeed; Position:= AutoSpeed;
  end;
  if G_Application.ExecDialog(New(P_OptionDlg,Init(@Self,'OPTIONS',@OptBuf)))=id_OK
  then begin
    for i:= 1 to C_MaxDisk do if OptBuf.Count[i] = bf_Checked then Count:= i;
    AutoSpeed:= OptBuf.Speed.Position;
    if Count <> DiskCount then begin DiskCount:= Count; Auto:= false; GameReset; end;
  end;
end;

procedure T_GameWindow.CMAuto(var V_Msg: TMessage);
begin AutoStart end;

procedure T_GameWindow.AutoStart;
begin
  GameReset;
  Auto:= true;
  IdleCount:= SpeedCount(AutoSpeed);
  Sequence^.Fill(DiskCount);
end;

procedure T_GameWindow.CMBegin(var V_Msg: TMessage);
begin GameReset; Auto:= false; end;

procedure T_GameWindow.CMGameExit(var V_Msg: TMessage);
begin CMExit(V_Msg) end;

procedure T_GameWindow.CMAbout;
begin
  MessageBox(HWindow,
  { text:  } 'Towers of Hanoi'#13#13 +
             'HANOWE v1.5 (MS-Windows)'#13 +
             '(c) J.R. Ferguson, 1996-2000'#13 +
             'j.r.ferguson@iname.com'#13 +
             'http://hello.to/ferguson',
  { title: } 'About HANOWE',
  { button:} mb_IconInformation or mb_OK);
end;

procedure T_GameWindow.CMRules;
begin
  MessageBox(HWindow,
  { text:  } 'Move all disks from left to right.'#13#13 +
             'You can only move one disk at a time.'#13 +
             'You may not place a disk on top of one that is smaller.',
  { title: } 'Rules of the Game',
  { button:} mb_OK);
end;

procedure T_GameWindow.MoveDragDisk(V_X, V_Y: integer);
begin
  with DragDisk^ do begin
    Draw(Context, DiskHidePen);
    Move(V_X, V_Y);
    Draw(Context, DiskShowPen);
  end;
  Draw(Context);
end;

procedure T_GameWindow.WMLButtonDown(var V_Msg: TMessage);
var
  t     : T_TowerInd;
  Touch : boolean;
begin if not Auto and (DragDisk = nil) then begin
  t:= 0; Touch := false;
  while not Touch and (t < C_MaxTower) do begin
    Inc(t);
    Touch:= Tower[t]^.SrcHit(integer(V_Msg.LParamLo),integer(V_Msg.LParamHi));
  end;
  if Touch then begin
    DragDisk := Tower[t]^.GetTop;
    if DragDisk <> nil then begin
      SetCapture(HWindow);
      Context  := GetDC(HWindow);
      MoveDragDisk(integer(V_Msg.LParamLo),integer(V_Msg.LParamHi));
    end;
  end;
end end;

procedure T_GameWindow.WMMouseMove(var V_Msg: TMessage);
begin
  if not Auto and (DragDisk <> nil) then
    MoveDragDisk(integer(V_Msg.LParamLo),integer(V_Msg.LParamHi));
end;

procedure T_GameWindow.WMLButtonUp  (var V_Msg: TMessage);
var
  t     : T_TowerInd;
  Touch : boolean;
begin if not Auto and (DragDisk <> nil) then begin
  DragDisk^.Draw(Context, DiskHidePen);
  t:= 0; Touch := false;
  while not Touch and (t < C_MaxTower) do begin
    Inc(t);
    Touch:= Tower[t]^.DstHit(integer(V_Msg.LParamLo),integer(V_Msg.LParamHi));
  end;
  if not (Touch and Tower[t]^.Connect(DragDisk)) then DragDisk^.Tower^.Connect(DragDisk);
  DragDisk:= nil;
  Draw(Context);
  ReleaseCapture;
  ReleaseDC(HWindow, Context);
  if (Tower[1]^.Top = nil) and (Tower[2]^.Top = nil) then begin
    MessageBox(HWindow,'Very good','Done',mb_IconExclamation or mb_OK);
    GameReset;
  end;
end end;

function T_GameWindow.IdleAction: boolean;
begin
  if not Auto then IdleAction:= false
  else begin
    if IdleCount > 0  then begin Dec(IdleCount); IdleAction:= true; end
    else begin
      IdleCount:= SpeedCount(AutoSpeed);
      if AutoMove then IdleAction:= true
      else begin Auto:= false; IdleAction:= false; end;
    end;
  end;
end;

function T_GameWindow.AutoMove: boolean;
var src, dst: T_TowerInd;
begin
  if Sequence^.GetMove(src,dst) then begin
    Context := GetDC(HWindow);
    DragDisk:= Tower[src]^.GetTop;
    DragDisk^.Draw(Context, DiskHidePen);
    Tower[dst]^.Connect(DragDisk);
    DragDisk:= nil;
    Draw(Context);
    ReleaseDC(HWindow, Context);
    AutoMove:= true;
  end
  else begin
    if MessageBox(HWindow,'Show again?','Done',mb_IconQuestion or mb_YesNo) = IdYes
    then begin AutoStart; AutoMove:= true; end
    else begin GameReset; Auto:= false; AutoMove:= false; end;
  end;
end;


{ --- T_Application methods --- }

procedure T_Application.InitMainWindow;
begin MainWindow:= New(P_GameWindow,Init(nil,'Towers of Hanoi')); end;

procedure   T_Application.InitInstance;
begin
  inherited InitInstance;
  HAccTable:= LoadAccelerators(HInstance,'ACC_KEYS');
end;

function T_Application.IdleAction: boolean;
begin IdleAction:= P_GameWindow(MainWindow)^.IdleAction; end;

{ --- Main program --- }

begin
  G_Application.Init('HANOWE');
  G_Application.Run;
  G_Application.Done;
end.
