#include "keyboard.h"

#include "terminal.h"
#include "beeper.h"

static unsigned char TextMoveBuffer[Columns*2u];

#define mygettext_line(x,y,count, target) \
    mygettext(FP_OFF(VidMem)+VidMemBytePos(x,y), count, target)

#define myfillbox_line(x,y,count,model) \
    myfilltext(FP_OFF(VidMem)+VidMemBytePos(x,y), count, model)

#define myfillbox_lines(y,nlines,model) \
    myfilltext_lines(FP_OFF(VidMem), y, nlines, model)

void termwindow::ResetAttr()
{
    intensity = underline = blink = reverse = 0;
    bgc = 0;
    fgc = 7;
}
void termwindow::BuildAttr()
{
    attr = (bgc<<4) | fgc;
    if(intensity==-1)attr = (attr&0xF0) | 8;
    if(reverse)attr = (attr & 0x88) | (((attr >> 4) | (attr << 4)) & 0x77);
    if(intensity==1)attr |= 8;
    if(blink)attr |= 0x80;
    textattr(attr);
    filler = (attr << 8) | ' ';
}
void termwindow::Reset()
{
    VidMem =    ((unsigned char* _far) 0xB8000000UL) +
       page * (*(unsigned short* _far) 0x0040004CUL);
    Cursor = (short _far*) VidMem;
    cx = cy = top = 0;
    bottom = Rows-1;

    g0set = 0; g1set = 1; activeset = 0; translate = Translate[g0set];
    utfmode = 0; utflength = 0; utfvalue = 0;
    state = ESnormal;
    ques = 0;
    Reset256Color();
    ResetAttr();
    BuildAttr();
    csi_J(2);
}

void termwindow::Lf()
{
    if(cy >= bottom)
    {
        /* scroll the window up */
        yscroll_up(top, bottom, 1);
    }
    else
    {
        ++cy;
        Cursor += Columns;
    }
}

void termwindow::Ri()
{
    if(cy <= top)
    {
        /* scroll the window down */
        yscroll_down(top, bottom, 1);
    }
    else
    {
        --cy;
        Cursor -= Columns;
    }
}

void termwindow::yscroll_down(unsigned y1, unsigned y2, int amount) const
{
    unsigned hei = y2-y1+1;
    if(amount > hei) amount = hei;
    mymovetext(FP_OFF(VidMem), y1, y2-amount, y1+amount);
    myfillbox_lines(y1, amount, filler);
}

void termwindow::yscroll_up(unsigned y1, unsigned y2, int amount) const
{
    unsigned hei = y2-y1+1;
    if(amount > hei) amount = hei;
    mymovetext(FP_OFF(VidMem), y1+amount, y2, y1);
    myfillbox_lines(y2-amount+1, amount, filler);
}

void termwindow::csi_at(register unsigned c) const
{
    // insert c spaces at cursor
    if(cx + c > Columns) c = Columns-cx;
    if(c == 0) return;
    unsigned remain = Columns - (cx+c);
    mygettext_line(cx,cy, remain, TextMoveBuffer);
    myfillbox_line(cx,cy, c, filler);
    myputtext_line(cx+c,cy, remain, TextMoveBuffer);
}

void termwindow::csi_X(register unsigned c) const
{
    // write c spaces at cursor (overwrite)
    myfillbox_line(cx,cy+top, c, filler);
}

void termwindow::csi_P(register unsigned c) const
{
    // insert c black holes at cursor (eat c characters
    // and scroll line horizontally to left)
    if(cx + c > Columns) c = Columns-cx;
    if(c == 0) return;
    unsigned remain = Columns - (cx+c);
    mygettext_line(Columns-remain,cy, remain, TextMoveBuffer);
    myfillbox_line(Columns-c,cy, c, filler);
    myputtext_line(cx,cy, remain, TextMoveBuffer);
}

void termwindow::csi_J(register unsigned c) const
{
    switch(c)
    {
        case 0: // erase from cursor to end of display
            csi_K(0);
            if(cy < Rows-1)
                myfillbox_lines(cy+1, Rows-cy-1, filler);
            break;
        case 1: // erase from start to cursor
            if(cy > 0) myfillbox_lines(0, cy, filler);
            csi_K(1);
            break;
        case 2: // erase whole display
            myfillbox_lines(0, Rows, filler);
            break;
    }
}

void termwindow::csi_K(register unsigned c) const
{
    // 0: erase from cursor to end of line
    // 1: erase from start of line to cursor
    // 2: erase whole line
    switch(c)
    {
        case 0: myfillbox_line(cx,cy, Columns-cx, filler); break;
        case 1: myfillbox_line(0, cy, cx+1,       filler); break;
        case 2: myfillbox_line(0, cy, Columns,    filler);
    }
}


void termwindow::Write(const char* const s, const unsigned length)
{
    // This is an indirect ripoff from
    // /usr/src/linux/drivers/char/console.c
    unsigned a, b=length;
    for(a=0; a<b; ++a)
    {
        int c = (unsigned char) s[a];
        switch(c)
        {
            case 7: BeepOn(); goto handled;
            case 8: ScrollFix(); if(cx>0) { --cx; --Cursor; } goto handled;
            case 9: ScrollFix(); cx += 8 - (cx & 7); FixCoord(); goto handled;
            case 10: case 11: case 12: Linefeed: ScrollFix();
            {
                if(cy == bottom)
                {
                    /* If the pending buffer has a few more linefeeds, do them
                     * at once to minimize the number of whole-screen scrolling
                     * operations done
                     */
                    int pending_linefeeds = 1;
                    for(unsigned c=a+1; c<b; ++c)
                        if(s[c] == 10 || s[c] == 11 || s[c] == 12)
                        { CalcLF:
                            pending_linefeeds += 1;
                            if(pending_linefeeds >= bottom-top+1) break;
                        }
                        else if(s[c] == '\033' || s[a] == '\245')
                        {
                            // If it's a color escape, ok; otherwise break
                            if(++c >= b) break;
                            if(s[c] == 'D' || s[c] == 'E') goto CalcLF;
                            if(s[c++] != '[') break;
                            if(c < b && s[c] == '?') ++c;
                            while(c < b && ((s[c]>='0' && s[c]<='9') || s[c]==';')) ++c;
                            if(c >= b) break;

                            if(s[c] != 'm' // color setting
                            && s[c] != 'C' // horizontal cursor positioning
                            && s[c] != 'D' // horizontal cursor positioning
                            && s[c] != 'K' // horizontal line clearing
                            && s[c] != 'G' // horizontal cursor positioning
                              ) break;
                        }
                        else {}
                    /* Note: ircII or GNU screen seems to use
                     * a \33[H\33[47B sequence to reposition
                     * the cursor right before each linefeed.
                     * Without changing the sequence, it cannot
                     * be optimized like regular LFs.
                     */
                    yscroll_up(top, bottom, pending_linefeeds);
                    cy -= pending_linefeeds-1;
                    if(cy < top) cy = top;
                    RecalculateCursor();
                    goto handled;
                }
                Lf();
                goto handled;
            }
            case 13: Cursor -= cx; cx=0; goto handled;
            case 14: activeset = 1; translate = Translate[g1set]; goto handled;
            case 15: activeset = 0; translate = Translate[g0set]; goto handled;
            case 24: case 26: state = ESnormal; goto handled;
            case 27: state = ESescnext; break;
            case 127: /* del - ignore */ goto handled;
            case 128+27: state = ESesc; break;
        }
        switch(state)
        {
            case ESescnext:
                state = ESesc;
                break;
            case ESnormal:
                if(utfmode)
                {
                    /* TODO: Parse UTF8 sequence */
                    ScrollFix();
                    *Cursor++ = (attr << 8) | translate[c];
                    ++cx;
                }
                else
                {
                    ScrollFix();
                    *Cursor++ = (attr << 8) | translate[c];
                    ++cx;
                }
                break;
            case ESesc:
                state = ESnormal;
                switch(c)
                {
                    case '[': state = ESsquare; break;
                    //case ']': state = ESnonstd; break;
                    case '%': state = ESpercent; break;
                    case 'E': Cursor -= cx; cx=0; goto Linefeed;
                    case 'M': Ri(); break;
                    case 'D': goto Linefeed;
                    case 'Z': EchoBack("\33[?6c", 5); break;
                    case '(': state = ESsetG0; break;
                    case ')': state = ESsetG1; break;
                    case '#': state = EShash; break;
                    case '7': save_cur(); break;
                    case '8': restore_cur(); break;
                    case 'c': Reset(); break;
                }
                break;
            case ESsquare:
                par.clear(); par.push_back(0);
                state = ESgetpars;
                if(c=='[') { state = ESignore; break; }
                if(c=='?') { ques=1; break; }
                ques = 0;
                /* fallthru */
            case ESgetpars:
                if(c==';') { par.push_back(0); break; }
                if(c>='0' && c<='9') { par[par.size()-1] = par[par.size()-1]*10 + c-'0'; break; }
                state = ESgotpars;
                /* fallthru */
            case ESgotpars:
                state = ESnormal;
                if(ques && (c=='c' /* cursor type */
                         || c=='m' /* complement mask */))
                {
                    /* UNIMPLEMENTED */
                    break;
                }

                switch(c)
                {
                    case 'h': /* misc modes on */ break;
                    case 'l':
                        /* misc modes off */
                        /* esc[?3;4l = monitor off, insert mode off */
                        /* esc>      = numeric keypad off */
                        /* UNIMPLEMENTED */
                        break;
                    case 'n':
                        if(ques) break;
                        if(par[0]==5) EchoBack("\033[0n", 4);
                        else if(par[0]==6) { char Buf[16]=""; // shut up BC warning
                            EchoBack(Buf, sprintf(Buf, "\33[%u;%uR",cy+1,cx+1)); }
                        break;
                    case 'G': case '`': if(par[0])--par[0]; cx=par[0]; goto cmov;
                    case 'd':           if(par[0])--par[0]; cy=par[0]; goto cmov;
                    case 'F': cx=0; // PASSTHRU TO 'A'
                    case 'A': if(!par[0])par[0]=1; cy-=par[0]; goto cmov;
                    case 'E': cx=0; // PASSTHRU TO 'B'
                    case 'B': if(!par[0])par[0]=1; cy+=par[0]; goto cmov;
                    case 'C': if(!par[0])par[0]=1; cx+=par[0]; goto cmov;
                    case 'D': if(!par[0])par[0]=1; cx-=par[0];
                              cmov: FixCoord(); break;
                    case 'H': case 'f': par.resize(2);
                        if(par[0])--par[0];
                        if(par[1])--par[1];
                        cx=par[1]; cy=par[0];
                        RecalculateCursor();
                        break;
                    case 'J': csi_J(par[0]); break;
                    case 'K': csi_K(par[0]); break;
                    case 'L': csi_L(par[0]); break;
                    case 'M': csi_M(par[0]); break;
                    case 'P': csi_P(par[0]); break;
                    case 'X': csi_X(par[0]); break;
                    case '@': csi_at(par[0]); break;
                    case 'c': if(!par[0])EchoBack("\33[?6c", 5); break;
                    case 'g': /* set tab stops UNIMPLEMENTED */ break;
                    case 'q': /* set leds UNIMPLEMENTED */ break;
                    case 'm':
                    {
                        int mode256 = 0;
                        for(unsigned a=0; a<par.size(); ++a)
                            switch(par[a])
                            {
                                case 0: mode256 = 0; ResetAttr(); break;
                                case 1: intensity = 1; break;
                                case 2: intensity = -1; break;
                                case 4: underline = 1; break;
                                case 5:
                                    if(!mode256) blink=1;
                                    else if(mode256==1) fgc = Find256Color(par[++a]);
                                    else if(mode256==2) bgc = Find256Color(par[++a]);
                                    break;
                                case 7: reverse = 1; break;
                                case 21: case 22: intensity = 0; break;
                                case 24: underline = 0; break;
                                case 25: blink = 0; break;
                                case 27: reverse = 0; break;
                                case 38: mode256 = 1; break;
                                case 39: underline = 0; fgc = 7; break;
                                case 48: mode256 = 2; break;
                                case 49: bgc = 0; break;
                                default:
                                    static const unsigned char swap[8] = {0,4,2,6,1,5,3,7};
                                    /**/ if(par[a]>=30 && par[a]<=37) fgc = swap[par[a]-30];
                                    else if(par[a]>=40 && par[a]<=47) bgc = swap[par[a]-40];
                                    /**/ if(par[a]>=90 && par[a]<=97) fgc = 8|swap[par[a]-90];
                                    else if(par[a]>=100&& par[a]<=107)bgc = 8|swap[par[a]-100];
                            }
                        BuildAttr();
                        break;
                    }
                    case 'r': par.resize(2);
                        if(!par[0])par[0]=1;
                        if(!par[1])par[1]=Rows;
                        if(par[0]<par[1] && par[1]<=Rows)
                        {
                            top=par[0]-1, bottom=par[1]-1;
                            cx=0; cy=top;
                            RecalculateCursor();
                        }
                        break;
                    case 's': save_cur(); break;
                    case 'u': restore_cur(); break;
                }
                break;
            case EShash:
                state = ESnormal;
                if(c == '8') { /* clear screen with 'E' */ break; }
                break;
            case ESignore:
                state = ESnormal;
                break;
            case ESsetG0:
                if(c == '0') g0set = 1;
                else if(c == 'B') g0set = 0;
                else if(c == 'U') g0set = 2;
                else if(c == 'K') g0set = 3;
                if(activeset == 0) translate = Translate[g0set];
                state = ESnormal;
                break;
            case ESsetG1:
                if(c == '0') g1set = 1;
                else if(c == 'B') g1set = 0;
                else if(c == 'U') g1set = 2;
                else if(c == 'K') g1set = 3;
                if(activeset == 1) translate = Translate[g1set];
                state = ESnormal;
                break;
            case ESpercent:
                state = ESnormal;
                if(c == '@') utfmode = 0;
                else if(c == 'G' || c == '8') utfmode = 1;
                break;
        }
     handled:;
    }

    if((cx+1 != Columns
     || cy+1 != Rows)
    )
    {
        mygotoxy(page, cx,cy);
    }
}

void termwindow::EchoBack(const char* buffer, unsigned size)
{
    //Write(buffer, size); // DEBUGGING

    while(size > 0 && OutBufferLength < OutBufferCapacity)
    {
        OutBuffer[OutBufferHead++] = *buffer++;
        if(OutBufferHead == OutBufferCapacity) OutBufferHead = 0;
        --size;
        ++OutBufferLength;
    }
}

void termwindow::save_cur()
{
    backup.cx = cx;
    backup.cy = cy;
    backup.i = intensity;
    backup.u = underline;
    backup.b = blink;
    backup.r = reverse;
    backup.f = fgc;
    backup.g = bgc;
    backup.top = top;
    backup.bottom = bottom;
}

void termwindow::restore_cur()
{
    cx = backup.cx;
    cy = backup.cy;
    intensity = backup.i;
    underline = backup.u;
    blink = backup.b;
    reverse = backup.r;
    fgc = backup.f;
    bgc = backup.g;
    top = backup.top;
    bottom = backup.bottom;
    BuildAttr();
    RecalculateCursor();
}
