; Cute Mouse Driver - a tiny mouse driver
; Copyright (c) 1997-2000 Nagy Daniel <nagyd@almos.vein.hu>
;
; This program is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or
; (at your option) any later version.
;
; This program is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with this program; if not, write to the Free Software
; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;

WARN

j		equ	jmp short

movSeg		macro	dest,src
		push	src
		pop	dest
	endm

saveFAR		macro	addr,segm,offs
		mov	word ptr addr[0],offs
		mov	word ptr addr[2],segm
	endm

PS2serv		macro	serv,errlabel
		mov	ax,serv
		int	15h
	IFNB <errlabel>
		jc	errlabel
		or	ah,ah
		jne	errlabel
	ENDIF
	endm

PrintS		macro	addr
	IFNB <addr>
	 IFDIFI <addr>,<dx>
		mov	dx,offset addr
	 ENDIF
	ENDIF
		mov	ah,9
		int	21h
	endm

POINT		struc
  X		dw	0
  Y		dw	0
	ends

driverversion	equ	626h		; Microsoft driver version

_ARG_ES_	equ	word ptr [bp]
_ARG_DI_	equ	word ptr [bp+2]
_ARG_SI_	equ	word ptr [bp+4]
_ARG_BP_	equ	word ptr [bp+6]
_ARG_BX_	equ	word ptr [bp+8]
_ARG_DX_	equ	word ptr [bp+10]
_ARG_CX_	equ	word ptr [bp+12]
_ARG_AX_	equ	word ptr [bp+14]
_ARG_DS_	equ	word ptr [bp+16]

.model tiny				; it is a COM file
.code
		org	100h
start:		jmp	real_start

; UNINITIALIZED DATA 
;!!! WARNING: don't modify uninitialized data in initialization part,
;	      because them not copied to new TSR

		org	4Eh		; reuse part of PSP

;----- application state begins here -----
;!!! WARNING: variables order between StartSaveArea and LenDefArea must be fixed !!!

		evendata
StartSaveArea = $

mickey8		POINT	?		; mickeys per 8 pixel ratios
;;+*doublespeed	dw	?		; double-speed threshold (mickeys/sec)
startscan	dw	?		; screen mask/cursor start scanline
endscan		dw	?		; cursor mask/cursor end scanline
;---------- hotspot, screenmask and cursormask must follow as is
hotspot		POINT	?		; hot spot in cursor bitmap
screenmask	db	2*16 dup (?)	; user defined screen mask
cursormask	db	2*16 dup (?)	; user defined cursor mask
cursorhidden	db	?		; 0 - cursor visible, else hidden
;;+*nolightpen?	db	?		; 0 - emulate light pen
		evendata
LenDefArea = $ - StartSaveArea		; initialized by softreset_21

rangemax	POINT	?		; horizontal/vertical range max
upleft		POINT	?		; upper left of update region
lowright	POINT	?		; lower right of update region
pos		POINT	?
granpos		POINT	?

		evendata
StartZeroArea = $

mickey		POINT	?
remain		POINT	?
		evendata
LenZeroArea1 = $ - StartZeroArea	; cleared by setpos_04

rangemin	POINT	?		; horizontal/vertical range min
		evendata
LenZeroArea2 = $ - StartZeroArea	; cleared by setupvideo

cursortype	db	?		; 0 - software, else hardware
callmask	db	?		; user interrupt routine call mask
buttstatus	dw	?		; buttons status
BUTTLASTSTATE	struc
  counter	dw	?
  lastrow	dw	?
  lastcol	dw	?
	ends
buttpress	BUTTLASTSTATE	?
		BUTTLASTSTATE	?
		BUTTLASTSTATE	?
		BUTTLASTSTATE	?
buttrelease	BUTTLASTSTATE	?
		BUTTLASTSTATE	?
		BUTTLASTSTATE	?
		BUTTLASTSTATE	?
		evendata
LenZeroArea3 = $ - StartZeroArea	; cleared by softreset_21
LenSaveArea = $ - StartSaveArea

;----- registers values for RIL -----
;!!! WARNING: registers order and RGROUPDEF contents must be fixed !!!
		evendata
StartVRegsArea = $

regs_SEQC	db	5 dup (?)
reg_MISC	db	?
regs_CRTC	db	25 dup (?)
regs_ATC	db	21 dup (?)
regs_GRC	db	9 dup (?)
reg_FC		db	?
reg_GPOS1	db	?
reg_GPOS2	db	?
		evendata
LenVRegsArea = $ - StartVRegsArea

		evendata
StartDefVRegsArea = $

def_SEQC	db	5 dup (?)
def_MISC	db	?
def_CRTC	db	25 dup (?)
def_ATC		db	21 dup (?)
def_GRC		db	9 dup (?)
def_FC		db	?
def_GPOS1	db	?
def_GPOS2	db	?
		evendata
ERRIF ($-StartDefVRegsArea ne LenVRegsArea) "VRegs area contents corrupted!"

;----- old interrupt vectors -----

		evendata
oldint33	dd	?		; old INT 33 handler address
oldIRQaddr	dd	?		; old IRQ handler address


; INITIALIZED DATA 

		evendata
StartTSRpart	label

StartDefArea = $
		POINT	{X=8,Y=16}	; mickey8
;;+*		dw	64		; doublespeed
		dw	77FFh,7700h	; startscan, endscan

		POINT	{X=0,Y=0}	; hotspot
		dw	03FFFh, 01FFFh, 00FFFh, 007FFh	; screenmask
		dw	003FFh, 001FFh, 000FFh, 0007Fh
		dw	0003Fh, 0001Fh, 001FFh, 000FFh
		dw	030FFh, 0F87Fh, 0F87Fh, 0FCFFh
		dw	00000h, 04000h, 06000h, 07000h	; cursormask
		dw	07800h, 07C00h, 07E00h, 07F00h
		dw	07F80h, 07C00h, 06C00h, 04600h
		dw	00600h, 00300h, 00300h, 00000h

		db	1		; cursorhidden
;;+*		db	0		; nolightpen?
		evendata
ERRIF ($-StartDefArea ne LenDefArea) "Defaults area contents corrupted!"

;----- driver and video state begins here -----

		evendata
userprocunlock	db	1		; 0 - user intr proc is in progress
videounlock	db	1		; 0 - screen manipulation is in progress

cursorptr	dd	0		; video memory pointer
cursordrawn	db	0		; 1 - restore screen data

videomode	db	?		; video mode number
granumask	POINT	{X=-1,Y=-1}
textbuf		label	word
buffer@		dd	?		; pointer to screen sprite copy

vdata1		dw	10h,1, 10h,3, 10h,4, 10h,5, 10h,8, 08h,2
VDATA1cnt	equ	($-vdata1)/4
vdata2		dw	10h,1, 10h,4, 10h,0105h, 10h,0FF08h, 08h,0F02h
VDATA2cnt	equ	($-vdata2)/4

;----- table of pointers to registers values for RIL -----
RGROUPDEF	struc
  port@		dw	?
  regs@		dw	?
  def@		dw	?
  regscnt	db	1
  rmodify?	db	0
RGROUPDEF	ends

		evendata
videoregs@	label
			; CRTC
	RGROUPDEF {port@=3D4h,regs@=regs_CRTC,def@=def_CRTC,regscnt=25}
			; Sequencer
	RGROUPDEF {port@=3C4h,regs@=regs_SEQC,def@=def_SEQC,regscnt=5}
			; Graphics controller
	RGROUPDEF {port@=3CEh,regs@=regs_GRC, def@=def_GRC, regscnt=9}
			; VGA attrib controller
	RGROUPDEF {port@=3C0h,regs@=regs_ATC, def@=def_ATC, regscnt=20}
			; VGA misc output and input
	RGROUPDEF {port@=3C2h,regs@=reg_MISC, def@=def_MISC}
			; Feature Control
	RGROUPDEF {port@=3DAh,regs@=reg_FC,   def@=def_FC}
			; Graphics 1 Position
	RGROUPDEF {port@=3CCh,regs@=reg_GPOS1,def@=def_GPOS1}
			; Graphics 2 Position
	RGROUPDEF {port@=3CAh,regs@=reg_GPOS2,def@=def_GPOS2}


; IRQ HANDLERS 

;

IRQhandler	proc	far
		;cld
		push	ds ax cx dx bx bp si di es
		movSeg	ds,cs
IRQproc		db	0EBh			; if PS/2 "j PS2proc"
		db	PS2proc-IRQproc-2	; else "mov al,20h"
		out	20h,al			; {20h} end of interrupt

		db	0BAh			; MOV DX,word
IO_address	dw	?			; COM port IO address
		push	dx
		add	dx,5
		in	al,dx			; {3FDh} check for overrun
		xchg	bx,ax			; OPTIMIZE: instead MOV BL,AL
		pop	dx
		in	al,dx			; {3F8h} flush receive buffer
		db	0B9h			; MOV CX,word
IOdone		db	?,0			; processed bytes counter

		test	bl,2
		jz	@@nooverrun		; jump if no overrun occured
		mov	[IOdone],ch		; zero counter
		j	@@exitIRQ

@@nooverrun:	shr	bl,1
		jnc	@@exitIRQ		; jump if data not ready
		db	0E8h			; CALL NEAR to mouseproc
mouseproc	dw	MSMproc-mouseproc-2

@@exitIRQ:	jmp	@rethandler
IRQhandler	endp

;
;!!! WARNING: buffer for copy of screen sprite when serial protocols

bufferSERIAL	label				; requires 3*16 bytes

;
;				Enable PS/2
;
;
; In:	none
; Out:	none
; Use:	none
; Modf:	BX, ES
; Call:	none
;
enablePS2	proc
		movSeg	es,cs
		mov	bx,offset IRQhandler
@PS2set:	PS2serv	0C207h			; es:bx=ptr to handler
		mov	bh,1			; set mouse on
		PS2serv	0C200h
		ret
enablePS2	endp

;
;				Disable PS/2
;
;
; In:	none
; Out:	none
; Use:	none
; Modf:	BX, ES
; Call:	none
;
disablePS2	proc
		xor	bx,bx			; set mouse off
		mov	es,bx
;;*		PS2serv	0C200h
;;*		PS2serv	0C207h			; es:bx=ptr to handler
;;*		ret
		j	@PS2set
disablePS2	endp

;

PS2proc		proc	far
		mov	bp,sp
		mov	cx,[bp+24]		; Y
		mov	bx,[bp+26]		; X
		mov	ax,[bp+28]		; Status
		neg	cx			; reverse Y movement
		test	al,20h			; check Y sign bit
		jz	@@noyneg
		mov	ch,0
@@noyneg:	test	al,10h			; check X sign bit
		jz	@@noxneg
		dec	bh
@@noxneg:	call	mouseupdate
		pop	es di si bp bx dx cx ax ds
                retf
PS2proc		endp
ERRIF ($-bufferSERIAL lt 3*16) "PS/2 handler too small for buffer!"

;
;!!! WARNING: buffer for copy of screen sprite when PS2 protocol

bufferPS2	label				; requires 3*16 bytes

;
;			Enable serial interrupt in PIC
;
;
; In:	none
; Out:	none
; Use:	IO_address
; Modf:	AX, DX, SI, IOdone, MSLTmiddle
; Call:	INT 21/25
;
enableCOM	proc
;---------- set new IRQ handler
		db	0B8h			; MOV AX,word
IRQintnum	db	?,25h			; INT number of selected IRQ
		mov	dx,offset IRQhandler
		int	21h			; set INT in DS:DX

;---------- set communication parameters (speed, parity, etc.)
		mov	si,[IO_address]
		mov	dx,si
		add	dx,3
		mov	al,80h
		out	dx,al			; {3FBh} set DLAB on

		xchg	dx,si
		mov	ax,96
		out	dx,ax			; {3F8h},{3F9h} 1200 baud

		xchg	dx,si
		db	0B0h			; MOV AL,byte
COMLCR		db	2
		out	dx,al			; {3FBh} DLAB off, no parity,
						;  stop=1, length=7/8
		inc	dx
		mov	al,0Bh
		out	dx,al			; {3FCh} DTR/RTS/OUT2 on

		inc	dx			; {3FDh} read LSR thus
		in	al,dx			;  clearing error bits
		mov	dx,si
		in	al,dx			; {3F8h} flush receive buffer

		inc	dx
		mov	al,1
		out	dx,al			; {3F9h} DR int enable

		dec	ax			; OPTIMIZE: instead MOV AL,0
		mov	[IOdone],al
		mov	[MSLTmiddle],al
;----------
		in	al,21h			; {21h} get PIC mask
		db	24h			; AND AL,byte
notPICstate	db	?			; clear bit to enable interrupt
		out	21h,al			; enable serial interrupts
		ret
enableCOM	endp

;
;			Disable serial interrupt of PIC
;
;
; In:	none
; Out:	none
; Use:	IO_address, oldIRQaddr
; Modf:	AX, DX
; Call:	INT 21/25
;
disableCOM	proc
		in	al,21h			; {21h} get PIC mask
		db	0Ch			; OR AL,byte
PICstate	db	?			; set bit to disable interrupt
		out	21h,al			; disable serial interrupts
;----------
		mov	dx,[IO_address]
		add	dx,3
		mov	al,0
		out	dx,al			; {3FBh} set DLAB off

		dec	dx
		dec	dx
		;mov	al,0
		out	dx,al			; {3F9h} all interrupts off

		add	dx,3
		in	al,dx			; {3FCh} modem ctrl
		and	al,0F3h
		out	dx,al			; {3FCh} OUT2 off

;---------- restore old IRQ handler
		db	0B8h			; MOV AX,word
IRQintnum2	db	?,25h			; INT number of selected IRQ
		push	ds
		lds	dx,[oldIRQaddr]
		int	21h			; set INT in DS:DX
		pop	ds
		ret
disableCOM	endp

;
;		Process mouse bytes the Microsoft/Logitech way
;

MSLTproc	proc
		test	al,40h			; synchro check
		jz	@@MSLTbyte2		; jump if non first byte

		mov	[IOdone],1		; request next 2/3 bytes
		mov	[MSLTbyte1],al
MSLTCODE1	db	091h			; if LT/WM "xchg ax,cx" else "ret"
		sub	al,3			; first byte after 3 bytes?
		jnz	@@MSLTret
		mov	[MSLTmiddle],al		; release middle button
@@MSLTret:	ret

@@MSLTbyte2:	jcxz	@@MSLTret		; skip nonfirst byte at start
		inc	[IOdone]		; request next byte
		loop	@@MSLTbyte3
		mov	[MSLTbyteX],al		; store X increment LO
		ret

@@MSLTbyte3:	loop	@@LTbyte4
		xchg	cx,ax			; OPTIMIZE: instead MOV CL,AL
		db	0B4h			; MOV AH,byte
MSLTbyte1	db	?			; first byte of MS/LT protocol

		;mov	al,0			; CX was 0
		shr	ax,2			; bits 1/0 - X increment HI
		db	0Ch			; OR AL,byte
MSLTbyteX	db	?
		mov	bh,ah
		cbw
		xchg	bx,ax			; X increment

		mov	al,0
		shr	ax,2			; bits 3/2 - Y increment HI
		or	al,cl
		mov	cl,ah			; bits 5/4 - L/R buttons
		cbw
		xchg	cx,ax			; Y increment

		xor	al,[MSLTmiddle]
		and	al,3			; L/R buttons change mask
		mov	dl,al
		or	dl,bl			; zero if no L/R buttons
		or	dl,cl			; change and mouse not moved
MSLTCODE2	db	0EBh			; if MS3 "jnz" else "j"
		db	@@MSLTupdate-MSLTCODE2-2
		or	al,4			; empty event togles button
		j	@@MSLTupdate

@@LTbyte4:	mov	[IOdone],ch		; CH=0, request next 3/4 bytes
MSLTCODE3	db	0B1h,3			; if LT "mov cl,3" else
						; if WM "mov cl,2" else "ret"
		shr	al,cl			; (MS defines only 3 bytes)
		xor	al,[MSLTmiddle]
		and	al,4			; zero if state not changed
		jz	@@MSLTret
		xor	cx,cx
		xor	bx,bx
@@MSLTupdate:	db	034h			; XOR AL,byte
MSLTmiddle	db	?			; middle button for MS3/LT
		mov	[MSLTmiddle],al
		j	mouseupdate
MSLTproc	endp

;
;		Process mouse bytes the Mouse Systems way
;

MSMproc		proc
		jcxz	@@MSMbyte1
		cbw
		dec	cx
		jz	@@MSMbyte24
		dec	cx
		jz	@@MSMbyte3
		loop	@@MSMbyte5

@@MSMbyte24:	add	[MSMincrX],ax
@@MSMnext:	inc	[IOdone]		; request next byte
@@MSMret:	ret

@@MSMbyte1:	mov	[MSMbyte1],al		; save buttons state
		mov	[MSMincrX],cx		; CX=0
		and	al,not 7
		cmp	al,80h			; sync check
		;je	@@MSMnext
		;ret
		jne	@@MSMret

@@MSMbyte3:	mov	[MSMincrY],ax
		j	@@MSMnext

@@MSMbyte5:	mov	[IOdone],ch		; CH=0
		db	05h			; ADD AX,word
MSMincrY	dw	?
		xchg	cx,ax			; OPTIMIZE: instead MOV CX,AX
		neg	cx
		db	0BBh			; MOV BX,word
MSMincrX	dw	?
		db	0B0h			; MOV AL,byte
MSMbyte1	db	?			; first byte of MSM protocol
		not	al			; bits 2/1/0 - L/M/R buttons
		test	al,6			; check the L and M state
		jpe	@@MSMupdate		; skip if the same
		xor	al,6			; swap L and M buttons
@@MSMupdate:	;j	mouseupdate
MSMproc		endp
ERRIF ($-bufferPS2 lt 3*16) "Serial handler too small for buffer!"

;
;			Update mouse status
;
;
; In:	AL					(new buttons state)
;	BX					(X mouse movement)
;	CX					(Y mouse movement)
; Out:	none
; Use:	mickey, granpos, callmask
; Modf:	AX, CX, DX, BX, SI, DI, buttstatus, userprocunlock
; Call:	updateposition, updatebutton, showcursor
;
mouseupdate	proc
		db	024h			; AND AL,byte
buttonsmask	db	7
		test	al,3			; check the L and R buttons
LEFTHANDCODE	db	?			; JMP SHORT for PS2^LHAND/JPE
		db	@@updatebut-LEFTHANDCODE-2
		xor	al,3			; swap
@@updatebut:	push	ax

;---------- calculate new screen position
		xchg	ax,bx			; OPTIMIZE: instead MOV AX,BX
		db	0B2h			; MOV DL,byte
mresolution0	db	0
		mov	bx,offset X
		call	updateposition

		xchg	ax,cx
		db	0B2h			; MOV DL,byte
mresolution1	db	0
		mov	bl,offset Y		; OPTIMIZE: BL instead BX
		call	updateposition
		or	cl,al			; movement flag

;---------- calculate new buttons state
		pop	dx
		mov	dh,dl
		xchg	dl,byte ptr [buttstatus] ; DL=buttons old state
		xor	dh,dl			; DH=buttons change state
						; CL=unrolled change state
		xor	bx,bx			; buttpress array counter

		mov	al,2			; indicate that 1 is pressed
		call	updatebutton
		mov	al,8			; indicate that 2 is pressed
		call	updatebutton
		mov	al,20h			; indicate that 3 is pressed
		call	updatebutton

;---------- call User Defined Handler
		dec	[userprocunlock]
		jnz	@@updatedone		; exit if user proc running

		mov	al,cl
		cbw				; OPTIMIZE: instead MOV AH,0
		mov	bx,[buttstatus]
		mov	cx,[granpos.X]
		mov	dx,[granpos.Y]
		mov	si,[mickey.X]
		mov	di,[mickey.Y]
		sti
		and	al,[callmask]		; is there a user call mask?
		jz	@@updateshow		; exit if not
		push	ds
		db	9Ah			; CALL FAR to userproc
userproc	dd	?
		pop	ds
@@updateshow:	call	showcursor
;----------
@@updatedone:	cli
		inc	[userprocunlock]
		ret
mouseupdate	endp

;
; In:	AX					(mouse movement)
;	BX					(offset X/offset Y)
;	DL					(resolution)
; Out:	AL					(1 - mickey count changed)
; Use:	mickey8, rangemin, rangemax, granumask
; Modf:	AX, DX, mickey, remain, pos, granpos
; Call:	none
;
updateposition	proc
		or	ax,ax
		jz	@@uposret

;---------- resolute
		cmp	ax,3
		jb	@@setnewpos		; skip if movement in [0..+2]
		cmp	ax,-3
		ja	@@setnewpos		; skip if movement in [-2..-1]

		xchg	dx,ax
		or	al,al
		jnz	@@rescheck
		mov	ax,dx			; compute auto resolution
		sar	ax,2			; remain one fourth
		jge	@@rescut
		neg	ax			; ...from absolute value
@@rescut:	cmp	al,10
		jbe	@@rescheck
		mov	al,10			; ...but no more than 10
@@rescheck:	cbw
		cmp	al,1
		xchg	dx,ax
		jbe	@@setnewpos		; skip if resolution = 0 or 1

		cmp	ax,8
		jb	@@reslow		; jump if movement in [3..7]
		cmp	ax,-8
		jbe	@@resmul		; jump if move out of [-7..+7]
@@reslow:	mov	dl,2			; small move multiply by 2
@@resmul:	imul	dx			; multiply with resolution
;----------
@@setnewpos:	add	word ptr mickey[bx],ax
		shl	ax,3
		add	ax,word ptr remain[bx]
		cwd				; apply mickeys per 8 pixel
		idiv	word ptr mickey8[bx]	;  ratio to convert into
		mov	word ptr remain[bx],dx	;  position shift
		add	ax,word ptr pos[bx]
;----------
@savecutpos:	cmp	ax,word ptr rangemin[bx]
		jg	@@cuthi
		mov	ax,word ptr rangemin[bx]
@@cuthi:	cmp	ax,word ptr rangemax[bx]
		jl	@@savepos
		mov	ax,word ptr rangemax[bx]

@@savepos:	mov	word ptr pos[bx],ax	; new position
		and	al,byte ptr granumask[bx]
		mov	word ptr granpos[bx],ax	; new granulated position
		mov	al,1
@@uposret:	ret
updateposition	endp

;
;		Update one button status regs to new values
;
;
; In:	AL					(unrolled press bit mask)
;	CL					(unrolled buttons change state)
;	DL					(buttons old state)
;	DH					(buttons change state)
;	BX					(buttpress array counter)
; Out:	CL
;	DX					(shifted right state)
;	BX					(next offset)
; Use:	granpos
; Modf:	AX, SI, buttpress, buttrelease
; Call:	none
;
updatebutton	proc
		mov	si,offset buttpress
		shr	dl,1
		jnc	@@updset		; jump if was released
		shl	al,1			; indicate that they released
		mov	si,offset buttrelease
@@updset:	shr	dh,1
		jnc	@@updret		; jump if button unchanged
		or	cl,al
		inc	[si+bx].counter
		mov	ax,[granpos.Y]
		mov	[si+bx].lastrow,ax
		mov	ax,[granpos.X]
		mov	[si+bx].lastcol,ax
@@updret:	add	bx,SIZE BUTTLASTSTATE
		ret
updatebutton	endp

; END OF IRQ HANDLERS 


; INT 10 HANDLER 

		evendata
RILtable	dw	offset RIL_F0		; RIL functions
		dw	offset RIL_F1
		dw	offset RIL_F2
		dw	offset RIL_F3
		dw	offset RIL_F4
		dw	offset RIL_F5
		dw	offset RIL_F6
		dw	offset RIL_F7

int10handler	proc
		cld
		or	ah,ah			; set video mode?
		jz	@@setmodreq
		cmp	ah,0F0h			; RIL func requested?
		jb	@@jmpold10
		cmp	ah,0F7h
		jbe	@@RIL
		cmp	ah,0FAh
		je	@@RIL_FA
@@jmpold10:	db	0EAh			; JMP FAR to old INT 10 handler
oldint10	dd	?

;========== set video mode
@@setmodreq:	push	ax
		mov	al,2			; OPTIMIZE: AL instead AX
		int	33h			; hide mouse cursor
		pop	ax
		pushf
		call	cs:[oldint10]
		push	ds ax cx dx bx bp si di es
		movSeg	ds,cs
		mov	[cursorhidden],1	; normalize flag
		call	setupvideo
@@exitINT10:	jmp	@rethandler

;========== RIL
@@RIL:		push	ds ax cx dx bx bp si di es
		movSeg	ds,cs
		mov	bp,sp
		mov	al,ah
		and	ax,0Fh			;!!! AH must be 0 for RIL_*
		mov	si,ax
		shl	si,1
		call	RILtable[si]
		j	@@exitINT10
;----------
@@RIL_FA:	movSeg	es,cs			; RIL FA - Interrogate driver
		mov	bx,offset RILversion
		iret
int10handler	endp

;
; RIL F0 - Read one register
;
;
; In:	DX					(group index)
;	BX					(register #)
; Out:	BL					(value)
; Use:	videoregs@
; Modf:	AL, SI
; Call:	none
;
RIL_F0		proc
		mov	si,dx
		mov	si,videoregs@[si].regs@
		cmp	dx,20h
		jae	@@retBL			; jump if single register
		add	si,bx
@@retBL:	lodsb
		mov	byte ptr [_ARG_BX_],al
		ret
RIL_F0		endp

;
; RIL F1 - Write one register
;
;
; In:	DX					(group index)
;	BL					(value for single reg)
;	BL					(register # otherwise)
;	BH					(value otherwise)
; Out:	BL					(value)
; Use:	none
; Modf:	AX
; Call:	RILwrite
;
RIL_F1		proc
		mov	ah,bl
		cmp	dx,20h
		jae	RILwrite		; jump if single registers
		xchg	ax,bx			; OPTIMIZE: instead MOV AX,BX
		mov	byte ptr [_ARG_BX_],ah
		;j	RILwrite
RIL_F1		endp

;
; In:	DX					(group index)
;	AL					(register # for regs group)
;	AH					(value to write)
; Out:	none
; Use:	videoregs@
; Modf:	AL, DX, BX, DI
; Call:	RILoutAH, RILgroupwrite
;
RILwrite	proc
		xor	bx,bx
		mov	di,dx
		cmp	dx,20h
		mov	dx,videoregs@[di].port@
		mov	videoregs@[di].rmodify?,dl ; OPTIMIZE: DL instead 1
		mov	di,videoregs@[di].regs@
		jae	@@RWsingle		; jump if single register
		mov	bl,al
@@RWsingle:	mov	[di+bx],ah
		jae	RILoutAH
		;j	RILgroupwrite
RILwrite	endp

;
; In:	DX					(IO port)
;	AL					(register #)
;	AH					(value to write)
; Out:	none
; Use:	videoregs@
; Modf:	none
; Call:	none
;
RILgroupwrite	proc
		cmp	dl,0C0h
		je	@@writeATC		; jump if ATTR controller
		out	dx,ax
		ret
@@writeATC:	push	ax dx
		mov	dx,videoregs@[(SIZE RGROUPDEF)*5].port@
		in	al,dx			; {3DAh} force address mode
		pop	dx ax
		out	dx,al			; {3C0h} select ATC register
RILoutAH:	xchg	al,ah
		out	dx,al			; {3C0h} modify ATC register
		xchg	al,ah
		ret
RILgroupwrite	endp

;
; RIL F2 - Read register range
;
;
; In:	CH					(starting register #)
;	CL					(# of registers)
;	DX					(group index: 0,8,10h,18h)
;	ES:BX					(buffer, CL bytes size)
; Out:	none
; Use:	videoregs@
; Modf:	AX, CX, SI, DI
; Call:	none
;
RIL_F2		proc
		sti
		mov	di,bx
		mov	si,dx
		mov	si,videoregs@[si].regs@
		mov	al,ch
		;mov	ah,0
		add	si,ax
RILmemcopy:	mov	ch,0
		shr	cx,1
		rep	movsw
		adc	cx,cx
		rep	movsb
		ret
RIL_F2		endp

;
; RIL F3 - Write register range
;
;
; In:	CH					(starting register #)
;	CL					(# of registers, >0)
;	DX					(group index: 0,8,10h,18h)
;	ES:BX					(buffer, CL bytes size)
; Out:	none
; Use:	videoregs@
; Modf:	AX, CX, DX, BX, DI
; Call:	RILgroupwrite
;
RIL_F3		proc
		mov	di,dx
		mov	dx,videoregs@[di].port@
		mov	videoregs@[di].rmodify?,dl ; OPTIMIZE: DL instead 1
		mov	di,videoregs@[di].regs@
RILgrouploop:	xor	ax,ax
		xchg	al,ch
		add	di,ax
@@R3loop:	mov	ah,es:[bx]
		mov	[di],ah
		inc	bx
		inc	di
		call	RILgroupwrite
		inc	ax			; OPTIMIZE: AX instead AL
		loop	@@R3loop
		ret
RIL_F3		endp

;
; RIL F4 - Read register set
;
;
; In:	CX					(# of registers, >0)
;	ES:BX					(table of registers records)
; Out:	none
; Use:	videoregs@
; Modf:	AL, CX, BX, DI
; Call:	none
;
RIL_F4		proc
		sti
		mov	di,bx
@@R4loop:	mov	bx,es:[di]
		inc	di
		inc	di
		mov	bx,videoregs@[bx].regs@
		mov	al,es:[di]
		inc	di
		xlat
		stosb
		loop	@@R4loop
		ret
RIL_F4		endp

;
; RIL F5 - Write register set
;
;
; In:	CX					(# of registers, >0)
;	ES:BX					(table of registers records)
; Out:	none
; Use:	none
; Modf:	AX, CX, DX, SI
; Call:	RILwrite
;
RIL_F5		proc
		mov	si,bx
@@R5loop:	lods	word ptr es:[si]
		xchg	dx,ax			; OPTIMIZE: instead MOV DX,AX
		lods	word ptr es:[si]
		call	RILwrite
		loop	@@R5loop
		ret
RIL_F5		endp

;
; RIL F7 - Define registers default
;
;
; In:	DX					(group index)
;	ES:BX					(table of one-byte entries)
; Out:	none
; Use:	videoregs@
; Modf:	CL, SI, DI, ES, DS
; Call:	RILmemcopy
;
RIL_F7		proc
		sti
		mov	si,bx
		mov	di,dx
		mov	cl,videoregs@[di].regscnt
		mov	videoregs@[di].rmodify?,cl ; OPTIMIZE: CL instead 1
		mov	di,videoregs@[di].def@
		push	es ds
		pop	es ds
		j	RILmemcopy
RIL_F7		endp

;
; RIL F6 - Revert registers to default
;
;
; In:	none
; Out:	none
; Use:	videoregs@
; Modf:	AX, CX, DX, BX, SI, DI, ES
; Call:	RILgrouploop
;
RIL_F6		proc
		movSeg	es,ds
		mov	si,offset videoregs@+(SIZE RGROUPDEF)*8

@@R6loop:	sub	si,SIZE RGROUPDEF
		xor	cx,cx
		xchg	cl,[si].rmodify?
		jcxz	@@R6next

		mov	bx,[si].def@
		mov	di,[si].regs@
		mov	dx,[si].port@
		mov	cl,[si].regscnt
		dec	cx			; OPTIMIZE: CX instead CL
		jnz	@@R6group
		mov	al,[bx]
		stosb
		out	dx,al
		;j	@@R6next
		j	@@R6loop		; single regs handled first

@@R6group:	inc	cx			; OPTIMIZE: CX instead CL
		;mov	ch,0
		call	RILgrouploop

@@R6next:	cmp	si,offset videoregs@
		ja	@@R6loop
		ret
RIL_F6		endp

; END OF INT 10 HANDLER 


;
;			Draw mouse cursor
;

drawcursor	proc
		jz	@graphcursor		; jump if graphics mode
		call	gettxtoffset
		cmp	[cursortype],ch		; OPTIMIZE: CH instead 0
		jz	@@softcursor		; jump if software cursor

;========== draw hardware text mode cursor
		shr	di,1
		mov	dx,videoregs@[0].port@	; CRTC port

		mov	ax,di
		mov	ah,al
		mov	al,0Fh
		out	dx,ax			; cursor position lo

		xchg	ax,di			; OPTIMIZE: instead MOV AX,DI
		mov	al,0Eh
		out	dx,ax			; cursor position hi
		ret

;========== draw software text mode cursor
@@softcursor:	jcxz	@@drawtext		; jump if cursor not drawn
		cmp	di,word ptr cursorptr[0]
		je	@@drawtextdone		; exit if position not changed
		push	di
		call	restoretext
		pop	di

@@drawtext:	mov	ax,[granpos.Y]		; get cursor position
		mov	bx,[granpos.X]
		mov	si,8			; OPTIMIZE: instead -[granumask.Y]
		call	checkifseen
		jc	@@drawret		; exit if not seen

		mov	word ptr cursorptr[0],di
		mov	es,word ptr cursorptr[2]
		mov	ax,es:[di]		; save char under cursor
		mov	textbuf[0],ax
		and	ax,[startscan]
		xor	ax,[endscan]
		stosw				; draw to new position
		mov	textbuf[2],ax

@@drawtextdone:	inc	[cursordrawn]		; we have to restore later
@@drawret:	ret

;========== draw graphics mode cursor
@graphcursor:	; if [videomode] not in [4-6, 0Dh-13h] "ret" else "push cx"
		db	?
		call	updatevregs
		mov	bx,[granpos.X]
		sub	bx,[hotspot.X]
		and	bx,[granumask.X]	; virtual granulated X
		mov	ax,[granpos.Y]
		sub	ax,[hotspot.Y]		; virtual Y

		push	ax
		call	getgroffset
		pop	ax cx
; cx=old cursordrawn, bx=X, ax=Y, di=video row offset, si=nextrow

		jcxz	@@drawgraph		; jump if cursor not drawn
		cmp	bx,[cursorX]
		jne	@@restsprite
		db	081h,0FFh		; CMP DI,word
cursorrow	dw	?
		je	@@drawgrdone		; exit if position not changed

; bx=X, ax=Y, di=video row offset, si=nextrow
@@restsprite:	push	ax bx
		call	restoresprite
		pop	bx ax

;---------- check if cursor seen and not in update region
; bx=X, ax=Y, di=video row offset, si=nextrow
@@drawgraph:	mov	[cursorX],bx
		mov	[cursorY],ax
		mov	[cursorrow],di
		mov	[nextrow],si

		db	0B1h			; MOV CL,byte
bitmapshift	db	?			; mode 13h=1, 0Dh=4, other=3
		shr	bx,cl
		shl	bx,cl			; virtual screen aligned X
		mov	si,16			; cursor height
		call	checkifseen
		jc	@@drawgrret		; exit if not seen

;---------- precompute some sprite parameters
		push	ax			; push [cursorY]
		shr	dx,cl			; right X byte offset
		mov	ax,[scanline]
		cmp	dx,ax
		jbe	@@spritesize
		xchg	dx,ax			; DX=min(DX,scanline)
@@spritesize:	sar	bx,cl			; left X byte offset (signed)
		db	0B8h			; MOV AX,word
cursorX		dw	?
		sub	cl,3			; mode 0Dh=1
		sar	ax,cl			; sprite shift for non 13h modes
		neg	bx
		jge	@@savespritesz
		add	dx,bx
		sub	di,bx
		xor	bx,bx
		and	al,7			; shift in byte (X%8)
@@savespritesz:	mov	word ptr cursorptr[0],di
		mov	[spritewidth],dx
		neg	al
		add	al,8			; 8+|X| or 8-X%8
		cbw				; OPTIMIZE: instead MOV AH,0
		push	ax bx

;---------- save sprite and draw cursor at new cursor position
		mov	al,0D6h			; screen source
		call	copysprite		; save new sprite

		pop	si cx ax		; spriteshift,,[cursorY]
		les	di,[cursorptr]
		;mov	dx,[nextrow]
		;xor	bx,bx			; mask offset (2*16 bytes)

@@makeloop:	cmp	ax,[maxcoordY]
		jae	@@makenexty		; jump if Y > max || Y < 0

		push	ax dx bx di
		call	makerow
		pop	di bx dx ax

@@makenexty:	inc	ax
		add	di,dx
		xor	dx,[nextxor]
		inc	bx
		inc	bx
		cmp	bx,2*16
		jb	@@makeloop
;----------
@@drawgrdone:	inc	[cursordrawn]
@@drawgrret:	;j	restorevregs
drawcursor	endp

;
;		Restore graphics card video registers
;

restorevregs	proc
		; if [planar videomode 0Dh-12h] "push ds" else "ret"
		db	?
		pop	es
		mov	bx,offset vdata1
		mov	cx,VDATA1cnt
		j	@writevideo
restorevregs	endp

;
;		Save & update graphics card video registers
;

updatevregs	proc
		; if [planar videomode 0Dh-12h] "push ds" else "ret"
		db	?
		pop	es
		mov	bx,offset vdata1
		mov	cx,VDATA1cnt
		mov	ah,0F4h			; read register set
		int	10h

		mov	bx,offset vdata2
		mov	cx,VDATA2cnt
@writevideo:	mov	ah,0F5h			; write register set
		int	10h
		ret
updatevregs	endp

;
;			Restore old screen content
;

restorescreen	proc
		jcxz	@restret		; exit if cursor was not drawn
		jnz	restoretext		; jump if text mode
		call	updatevregs
		call	restoresprite
		j	restorevregs
restorescreen	endp

;

restoretext	proc
		les	di,[cursorptr]
		mov	ax,es:[di]
		cmp	ax,textbuf[2]
		jne	@restret
		mov	ax,textbuf[0]
		stosw				; restore old text char attrib
@restret:	ret
restoretext	endp

;

restoresprite	proc
		mov	al,0D7h			; screen destination
		;j	copysprite		; restore old sprite
restoresprite	endp

;
;		Copy screen sprite back and forth
;
;
; In:	AL					(0D6h-scr src./0D7h-scr dst.)
; Out:	DX = [nextrow]
;	BX = 0
; Use:	buffer@, cursorptr
; Modf:	AX, CX, ES
; Call:	none
;
copysprite	proc
		push	si di ds
		cmp	al,0D6h
		mov	NEXTOFFSCODE[1],al
		db	0B8h			; MOV AX,word
cursorY		dw	?			; cursor screen Y pos
		db	0BAh			; MOV DX,word
nextrow		dw	?			; next row offset
		mov	bx,16			; sprite height in rows
		les	di,[buffer@]
		lds	si,[cursorptr]
		je	@@copyloop
		push	ds es
		pop	ds es
		xchg	si,di

@@copyloop:	push	dx
		db	03Dh			; CMP AX,word
maxcoordY	dw	?			; screen height
		jae	@@copynexty		; jump if Y > max || Y < 0

		db	0B9h			; MOV CX,word
spritewidth	dw	?			; seen part of sprite in bytes
		sub	dx,cx
		rep	movsb

@@copynexty:	inc	ax
NEXTOFFSCODE	db	01h,?			; ADD SI,DX/ADD DI,DX
		pop	dx
		db	081h,0F2h		; XOR DX,word
nextxor		dw	?			; even times remain unchanged
		dec	bx
		jnz	@@copyloop
		pop	ds di si
		ret
copysprite	endp

;
;		Transform the cursor mask row to screen
;
;
; In:	BX					(mask offset)
;	CX					(sprite shift when non 13h modes)
;	SI					(sprite shift when mode 13h)
;	ES:DI					(video memory pointer)
; Out:	none
; Use:	screenmask, cursormask
; Modf:	AX, CX, DX, BX, SI, DI
; Call:	none
;
makerow		proc
		mov	dx,word ptr screenmask[bx]
		mov	bx,word ptr cursormask[bx]
		mov	ax,[spritewidth]
		cmp	[videomode],13h
		jne	makerowno13
;----------
		mov	cx,si
		shl	dx,cl
		shl	bx,cl
		xchg	cx,ax			; OPTIMIZE: instead MOV CX,AX

@@loopx13:	mov	al,0
		shl	dx,1
		jnc	@@chkcmask		; jump if most sign bit zero
		mov	al,es:[di]
@@chkcmask:	shl	bx,1
		jnc	@@putbyte13		; jump if most sign bit zero
		xor	al,0Fh
@@putbyte13:	stosb
		loop	@@loopx13
		ret

;---------- display cursor row in modes other than 13h
makerowno13:	push	cx
		xchg	si,ax			; OPTIMIZE: instead MOV SI,AX
		mov	ax,0FFh
@@maskshift:	stc
		rcl	dx,1
		rcl	al,1			; al:dh:dl shifted screenmask
		shl	bx,1
		rcl	ah,1			; ah:bh:bl shifted cursormask
		loop	@@maskshift
		xchg	dh,bl			; al:bl:dl - ah:bh:dh

@@loopxno13:	push	dx
		cmp	[videomode],0Dh		; is videomode 0Dh-12h?
		jae	@@planar		; jump if multiplanar mode
		and	al,es:[di]
		xor	al,ah
		stosb
		j	@@nextxno13

@@planar:	xchg	cx,ax			; OPTIMIZE: instead MOV CX,AX
		mov	dx,3CEh
		mov	ax,5			; set write mode
		out	dx,ax
		mov	ax,0803h		; data ANDed with latched data
		out	dx,ax
		xchg	es:[di],cl
		mov	ah,18h			; data XORed with latched data
		out	dx,ax
		xchg	es:[di],ch
		inc	di

@@nextxno13:	xchg	ax,bx			; OPTIMIZE: instead MOV AX,BX
		pop	bx
		dec	si
		jnz	@@loopxno13
		pop	cx
		ret
makerow		endp

;
;		Return graphic mode video memory offset
;
;
; In:	AX					(Y coordinate in pixels)
; Out:	DI					(video memory offset)
;	SI					(offset to next row)
; Use:	videomode
; Modf:	AX, DX, ES
; Call:	@getoffsret
;
getgroffset	proc
;
;4/5 (320x200x4)   byte offset = (y/2)*80 + (y%2)*2000h + (x*2)/8
;  6 (640x200x2)   byte offset = (y/2)*80 + (y%2)*2000h + x/8
;0Dh (320x200x16)  byte offset = y*40 + x/8, bit offset = 7 - (x % 8)
;0Eh (640x200x16)  byte offset = y*80 + x/8, bit offset = 7 - (x % 8)
;0Fh (640x350x4)   byte offset = y*80 + x/8, bit offset = 7 - (x % 8)
;10h (640x350x16)  byte offset = y*80 + x/8, bit offset = 7 - (x % 8)
;11h (640x480x2)   byte offset = y*80 + x/8, bit offset = 7 - (x % 8)
;12h (640x480x16)  byte offset = y*80 + x/8, bit offset = 7 - (x % 8)
;13h (320x200x256) byte offset = y*320 + x
;HGC (720x348x2)   byte offset = (y%4)*2000h + (y/4)*90 + x/8
;			bit offset = 7 - (x % 8)
;
		xor	di,di
		mov	es,di
		db	0BAh			; MOV DX,word
scanline	dw	?
		mov	si,dx			; [nextrow]
		cmp	byte ptr cursorptr[3],0A0h
		je	@getoffsret		; jump if not videomode 4-6
		mov	si,2000h
		sar	ax,1			; AX=Y/2
		jnc	@getoffsret
		mov	di,si			; DI=X/8+(Y%2)*2000h
		mov	si,-(2000h-80)
		j	@getoffsret
getgroffset	endp

;
;		Return text mode video memory offset
;
;
; In:	none
; Out:	DI					(video memory offset)
; Use:	0:44Ah, 0:44Eh, granpos
; Modf:	AX, DX, ES
; Call:	getpageoffset
;
gettxtoffset	proc
;
;0/1 (40x25) byte offset = (yc*40 + xc)*2 = yc*[0:44Ah]*2 + xc*2
;2/3 (80x25) byte offset = (yc*80 + xc)*2 = yc*[0:44Ah]*2 + xc*2
;  7 (80x25) byte offset = (yc*80 + xc)*2 = yc*[0:44Ah]*2 + xc*2
;
		xor	dx,dx
		mov	es,dx
		mov	di,[granpos.X]
		mov	al,[bitmapshift]
		dec	ax			; OPTIMIZE: AX instead AL
		xchg	cx,ax
		sar	di,cl			; X in characters*2
		xchg	cx,ax			; OPTIMIZE: instead MOV CX,AX
		mov	ax,[granpos.Y]
		shr	ax,1			; OPTIMIZE: instead DIV -[granumask.Y]/2
		shr	ax,1			; Y in characters*2
		mov	dx,es:[44Ah]		; screen width

@getoffsret:	imul	dx
		add	ax,es:[44Eh]		; add video page offset
		add	di,ax
		ret
gettxtoffset	endp

;
;		Check if cursor in update region
;

checkifseen	proc
		db	0BAh			; MOV DX,word
cursorwidth	db	?,0

;---------- check if cursor shape seen on the screen
		add	si,ax
		jle	@@retunseen
		add	dx,bx
		jle	@@retunseen
		cmp	ax,[maxcoordY]
		jge	@@retunseen
		cmp	bx,[maxcoordX]
		jge	@@retunseen

;---------- check if cursor shape not touches update region
		cmp	ax,[lowright.Y]
		jg	@@retseen
		cmp	bx,[lowright.X]
		jg	@@retseen
		cmp	si,[upleft.Y]
		jle	@@retseen
		cmp	dx,[upleft.X]
		jle	@@retseen

@@retunseen:	stc
		ret
@@retseen:	clc
		ret
checkifseen	endp


; INT 33 HANDLER SERVICES 

;

setupvideo	proc
		mov	si,LenZeroArea2/2	; clear area 2
ERRIF (LenZeroArea2 mod 2 ne 0) "LenZeroArea2 must be even!"
		j	@setvideo
setupvideo	endp

;
; 21 - Software reset
;
;
; In:	none
; Out:	[AX] = 21h/FFFFh			(not installed/installed)
;	[BX] = 2/3/FFFFh			(number of buttons)
; Use:	0:449h, 0:44Ah, 0:463h, 0:484h, 0:487h, 0:488h, 0:4A8h, shape
; Modf:	StartSaveArea, videomode, nextxor, buffer@, restorevregs, updatevregs,
;	@graphcursor, scanline, maxcoordX, maxcoordY, granumask, cursorptr,
;	bitmapshift, cursorwidth, rangemax, StartZeroArea
; Call:	hidecursor, @savecutpos
;
softreset_21	proc
		db	0B8h			; MOV AX,word
buttonscnt	db	3,0			; buttons count (2/3)
		mov	[_ARG_BX_],ax
		mov	[_ARG_AX_],0FFFFh

		movSeg	es,ds
		mov	si,offset StartDefArea
		mov	di,offset StartSaveArea
		mov	cx,LenDefArea/2
ERRIF (LenDefArea mod 2 ne 0) "LenDefArea must be even!"
		rep	movsw			; make default values
		call	hidecursor
		mov	si,LenZeroArea3/2	; clear area 3
ERRIF (LenZeroArea3 mod 2 ne 0) "LenZeroArea3 must be even!"

;---------- setup video regs values for current video mode
@setvideo:	push	si
		movSeg	es,ds
		xor	ax,ax
		mov	ds,ax
		mov	ax,ds:[463h]		; CRTC base I/O port address
		mov	es:videoregs@[0].port@,ax ; 3D4h/3B4h
		add	ax,6
		mov	es:videoregs@[(SIZE RGROUPDEF)*5].port@,ax
		mov	al,ds:[449h]		; current video mode
		and	al,not 80h
		push	ax

		cmp	al,11h			; VGA videomodes?
		jb	@@checkv1
		add	al,9
		j	@@getshift

@@checkv1:	cmp	al,0Fh			; 0F-10 videomodes?
		jb	@@checkv2
		test	byte ptr ds:[487h],60h	; check RAM size on adapter
		jz	@@getshift
		add	al,2
		j	@@getshift

@@checkv2:	cmp	al,4			; not color text modes?
		jae	@@getshift
		mov	ah,ds:[488h]		; get display combination
		and	ah,1Fh
		cmp	ah,3			; MDA+EGA/ECD?
		je	@@lines350
		cmp	ah,9			; EGA/ECD+MDA?
		jne	@@getshift
@@lines350:	add	al,13h

@@getshift:	lds	si,dword ptr ds:[4A8h]
		lds	si,dword ptr ds:[si]
		cbw				; OPTIMIZE: instead MOV AH,0
		mov	cl,6
		shl	ax,cl
		add	al,5			; OPTIMIZE: AL instead AX
		add	si,ax			; SI += AL*64+5

		mov	di,offset StartDefVRegsArea
		push	di			; copy default registers value
		mov	al,3
		stosb				; def_SEQ[0]=3
		mov	cx,50/2
		rep	movsw
		mov	al,0
		stosb				; def_ATC[20]=0, for VGA only
		movsw				; def_GRC (9 registers)
		movsw
		movsw
		movsw
		movsb
		stosw				; def_FC=0, def_GPOS1=0
		inc	ax			; OPTIMIZE: instead MOV AL,1
		stosb				; def_GPOS2=1

		movSeg	ds,es			; copy current registers value
		pop	si
		mov	di,offset StartVRegsArea
		mov	cx,LenVRegsArea/2
ERRIF (LenVRegsArea mod 2 ne 0) "LenVRegsArea must be even!"
		rep	movsw

		dec	ax			; OPTIMIZE: instead MOV AL,0
		mov	cx,8
		mov	di,offset videoregs@[0].rmodify?
@@setmodify:	stosb
		add	di,(SIZE RGROUPDEF)-1
		loop	@@setmodify

;---------- set parameters for current video mode
; mode	 seg   screen  cell scan planar pos/byte
;			    line
;  0	B800h  640x200 16x8   -    -	   -
;  1	B800h  640x200 16x8   -    -	   -
;  2	B800h  640x200	8x8   -    -	   -
;  3	B800h  640x200	8x8   -    -	   -
;  4	B800h  640x200	2x1  80   no	   8
;  5	B800h  640x200	2x1  80   no	   8
;  6	B800h  640x200	1x1  80   no	   8
;  7	B000h  640x200	8x8   -    -	   -
; 0Dh	A000h  320x200	2x1  40    +	  16
; 0Eh	A000h  640x200	1x1  80    +	   8
; 0Fh	A000h  640x350	1x1  80    +	   8
; 10h	A000h  640x350	1x1  80    +	   8
; 11h	A000h  640x480	1x1  80    +	   8
; 12h	A000h  640x480	1x1  80    +	   8
; 13h	A000h  320x200	2x1 320   no	   2
;
		pop	ax			; current video mode
; mode 0-3
		mov	bh,0B8h			; B800h: 0-3
		mov	cx,0304h		; 16x8: 0-1
		mov	dx,0FFFFh
		mov	di,200			; x200: 4-6, 0Dh-0Eh, 13h
		cmp	al,2
		jb	@@settext
		dec	cx			; 8x8: 2-3, 7
		cmp	al,4
		jb	@@settext
; mode 7
		cmp	al,7
		jne	@@checkgraph
		mov	bh,0B0h			; B000h: 7

@@settext:	mov	ch,1
		mov	dh,-8
		shl	dl,cl

		xor	ax,ax
		mov	es,ax
		mov	al,8
		mul	byte ptr es:[484h]	; screen height-1
		jz	@@screenw
		add	ax,8
		xchg	di,ax			; OPTIMIZE: instead MOV DI,AX

@@screenw:	mov	ax,es:[44Ah]		; screen width
		shl	ax,cl
		j	@@setcommon

; mode 4-6
@@checkgraph:	;mov	cx,0303h		; sprite: 3 bytes/row
		;mov	dx,0FFFFh		; 1x1: 6, 0Eh-12h
		;mov	bh,0B8h			; B800h: 4-6
		mov	bl,0C3h			; RET opcode
		push	bx
		;mov	di,200			; x200: 4-6, 0Dh-0Eh, 13h
		mov	si,2000h xor -(2000h-80) ; [nextxor] for modes 4-6
		;movSeg	es,ds
		db	0BBh			; MOV BX,word
BUFFADDR	dw	offset bufferPS2
		cmp	al,6
		je	@@setgraphics
		jb	@@set2x1
; mode 8-
		pop	bx
		mov	bh,0A0h			; A000h: 0Dh-
; for modes 0Dh-13h we'll be storing the cursor shape and the hided
; screen contents at free space in video memory to save driver memory
		mov	si,0A000h
		mov	es,si
		xor	si,si			; [nextxor] for modes >6
		cmp	al,13h
		ja	@@nonstandard
		jb	@@mode0812
; mode 13h
		push	bx
		mov	bx,0FA00h		; =320*200
		mov	cx,1000h		; sprite: 16 bytes/row
@@set320:	inc	cx			; OPTIMIZE: instead INC CL
@@set2x1:	dec	dx			; OPTIMIZE: instead MOV DL,-2
		j	@@setgraphics
; mode 8-0Dh
@@mode0812:	cmp	al,0Dh
		jb	@@nonstandard
		mov	bl,1Eh			; PUSH DS opcode
		push	bx
		mov	bx,3E82h		; 16002: 0Dh-0Eh
		je	@@set320
; mode 0Eh-12h
		cmp	al,0Fh
		jb	@@setgraphics
		mov	di,350			; x350: 0Fh-10h
		mov	bh,7Eh			; 32386: 0Fh-10h
		cmp	al,11h
		jb	@@setgraphics
		mov	di,480			; x480: 11h-12h
		mov	bh,9Eh			; 40578: 11h-12h

@@setgraphics:	mov	[videomode],al
		mov	[nextxor],si
		saveFAR	[buffer@],es,bx
		pop	ax			; multiplanar modes opcodes
		mov	byte ptr [restorevregs],al
		mov	byte ptr [updatevregs],al
		xchg	bx,ax			; OPTIMIZE: instead MOV BH,AH
		mov	al,51h			; PUSH CX opcode
		j	@@setgcommon

@@nonstandard:
;;+++++ for text modes: bh := 0B8h, dl := 0FFh, cl := 3, j @@settext
		;mov	dx,0FFFFh
		;mov	di,200
		mov	al,0C3h			; RET opcode

@@setgcommon:	mov	byte ptr [@graphcursor],al
		mov	ax,640			; [maxcoordX]
		shr	ax,cl
		mov	[scanline],ax		; screen line width in bytes
		shl	ax,cl

@@setcommon:	mov	[maxcoordX],ax
		mov	[maxcoordY],di
		mov	byte ptr [granumask.X],dl
		mov	byte ptr [granumask.Y],dh
		mov	byte ptr cursorptr[3],bh
		mov	[bitmapshift],cl	; log2(screen/memory ratio)
						;  (mode 13h=1, 0-1/0Dh=4, other=3)
		shl	ch,cl
		mov	[cursorwidth],ch
		pop	si

;---------- set ranges and center cursor (AX=maxcoordX, DI=maxcoordY)
		mov	cx,ax
		dec	ax
		mov	[rangemax.X],ax		; set right X range
		shr	cx,1			; X middle

		mov	dx,di
		dec	di
		mov	[rangemax.Y],di		; set lower Y range
		shr	dx,1			; Y middle

;---------- set cursor position (CX=X, DX=Y, SI=area size to clear)
@setpos:	cli
		movSeg	es,ds
		mov	di,offset StartZeroArea
		xchg	cx,si
		xor	ax,ax
		rep	stosw

		xchg	ax,dx			; OPTIMIZE: instead MOV AX,DX
		mov	bx,offset Y
		call	@savecutpos
		xchg	ax,si			; OPTIMIZE: instead MOV AX,SI
		mov	bl,offset X		; OPTIMIZE: BL instead BX
		jmp	@savecutpos
softreset_21	endp

;
; 1F - Disable mouse driver
;
;
; In:	none
; Out:	[AX] = 1Fh/FFFFh			(success/unsuccess)
;	[ES:BX]					(old int33 handler)
; Use:	oldint33, oldint10
; Modf:	AX, CX, DX, BX, ES, disabled?, cursorhidden
; Call:	INT 21/35, INT 21/25, disablePS2/disableCOM, hidecursor
;
disabledrv_1F	proc
		les	ax,[oldint33]
		mov	[_ARG_ES_],es
		mov	[_ARG_BX_],ax

		db	0E8h			; CALL NEAR to disableproc
disableproc	dw	disablePS2-disableproc-2

		mov	al,1
		cmp	[disabled?],al
		je	@@disabled
		mov	[cursorhidden],al	; normalize flag
		call	hidecursor

;---------- check if INT 33 or INT 10 were intercepted
;	    (handlers segments not equal to CS)
		mov	cx,cs
		mov	ax,3533h
		int	21h
		mov	dx,es
		cmp	dx,cx
		jne	@@disabled

		mov	al,10h			; OPTIMIZE: instead MOV AX,3510h
		int	21h
		mov	ax,es
		sub	ax,cx
		jne	@@disabled

		inc	ax			; OPTIMIZE: instead MOV AL,1
		mov	[disabled?],al

;---------- restore old INT 10 handler
		mov	ax,2510h
		lds	dx,[oldint10]
		int	21h
		ret
;----------
@@disabled:	mov	[_ARG_AX_],0FFFFh
		ret
disabledrv_1F	endp

;
; 00 - Reset driver and read status
;
;
; In:	none
; Out:	[AX] = 0/FFFFh				(not installed/installed)
;	[BX] = 2/3/FFFFh			(number of buttons)
; Use:	none
; Modf:	none
; Call:	softreset_21, enabledriver_20
;
resetdriver_00	proc
		call	softreset_21
		;j	enabledriver_20
resetdriver_00	endp

;
; 20 - Enable mouse driver
;
;
; In:	none
; Out:	[AX] = 20h/FFFFh			(success/unsuccess)
; Use:	none
; Modf:	AX, CX, DX, BX, ES, disabled?, oldint10
; Call:	INT 21/35, INT 21/25, setupvideo, enablePS2/enableCOM
;
enabledriver_20	proc
		cli
		xor	cx,cx
		xchg	cl,[disabled?]
		jcxz	@@enabled

;---------- set new INT 10 handler
		mov	ax,3510h
		int	21h			; get INT in ES:BX
		saveFAR	[oldint10],es,bx
		mov	ah,25h			; OPTIMIZE: instead MOV AX,2510H
		mov	dx,offset int10handler
		int	21h			; set INT in DS:DX
;----------
@@enabled:	call	setupvideo
		db	0E9h			; JMP NEAR to enableproc
enableproc	dw	enablePS2-enableproc-2
enabledriver_20	endp

;
; 03 - Get cursor position and buttons status
;
;
; In:	none
; Out:	[BX]					(buttons status)
;	[CX]					(X - column)
;	[DX]					(Y - row)
; Use:	buttstatus, granpos
; Modf:	AX, CX, DX
; Call:	none
;
status_03	proc
		mov	ax,[buttstatus]
		mov	cx,[granpos.X]
		mov	dx,[granpos.Y]
		j	@retBCDX
status_03	endp

;
; 05 - Get button press data
;
;
; In:	BX					(button number)
; Out:	[AX]					(buttons status)
;	[BX]					(press times)
;	[CX]					(last press X)
;	[DX]					(last press Y)
; Use:	buttpress
; Modf:	AX
; Call:	@retbuttstat
;
butpresdata_05	proc
		mov	ax,offset buttpress
		j	@retbuttstat
butpresdata_05	endp

;
; 06 - Get button release data
;
;
; In:	BX					(button number)
; Out:	[AX]					(buttons status)
;	[BX]					(release times)
;	[CX]					(last release X)
;	[DX]					(last release Y)
; Use:	butrelease, buttstatus
; Modf:	AX, CX, DX, BX
; Call:	none
;
buttreldata_06	proc
		mov	ax,offset buttrelease
@retbuttstat:	and	bx,7
ERRIF (6 ne SIZE BUTTLASTSTATE) "BUTTLASTSTATE structure size changed!"
		shl	bx,1
		add	ax,bx
		shl	bx,1
		add	bx,ax			; =AX+BX*SIZE BUTTLASTSTATE
		mov	ax,[buttstatus]
		mov	[_ARG_AX_],ax
		xor	ax,ax
		xchg	[bx.counter],ax
		mov	cx,[bx.lastcol]
		mov	dx,[bx.lastrow]
@retBCDX:	mov	[_ARG_CX_],cx
		mov	[_ARG_DX_],dx
@retBX:		mov	[_ARG_BX_],ax
		ret
buttreldata_06	endp

;
; 15 - Get driver storage requirements
;
;
; In:	none
; Out:	[BX]					(buffer size)
; Use:	LenSaveArea
; Modf:	AX
; Call:	none
;
storagereq_15	proc
		mov	ax,LenSaveArea
		j	@retBX
storagereq_15	endp

;
; 1B - Get mouse sensitivity
;
;
; In:	none
; Out:	[BX]				(number of mickeys per 8 pix
;	[CX]				 horizontally/vertically)
;	[DX]				(speed threshold in mickeys/second)
; Use:	mickey8, /doublespeed/
; Modf:	AX, CX, DX
; Call:	none
;
getsens_1B	proc
;;+*		mov	dx,[doublespeed]
		mov	dx,64
		mov	ax,[mickey8.X]
		mov	cx,[mickey8.Y]
		j	@retBCDX
getsens_1B	endp

;
; 1E - Get display page
;
;
; In:	none
; Out:	[BX]					(display page number)
; Use:	0:462h
; Modf:	AX, DS
; Call:	none
;
getpage_1E	proc
		xor	ax,ax
		mov	ds,ax
		mov	al,ds:[462h]
		j	@retBX
getpage_1E	endp

;
; 01 - Show mouse cursor
;
;
; In:	none
; Out:	none
; Use:	none
; Modf:	AX, lowright.Y, cursorhidden
; Call:	@showcursor, @showcheck
;
showcursor_01	proc
		mov	byte ptr lowright.Y[1],-1 ; make lowright.Y negative
		cmp	[cursorhidden],ah	; OPTIMIZE: AH instead 0
		jz	@showcursor		; jump if already shown
		dec	[cursorhidden]
		j	@showcheck
showcursor_01	endp

;
; 02 - Hide mouse cursor
;
;
; In:	none
; Out:	none
; Use:	none
; Modf:	cursorhidden
; Call:	none
;
hidecursor_02	proc
		inc	[cursorhidden]
		jnz	hidecursor		; jump if counter not overflow
		dec	[cursorhidden]

hidecursor:	mov	dx,offset restorescreen
		j	@cursorproc
hidecursor_02	endp

;
; 07 - Set horizontal cursor range
;
;
; In:	CX					(min X)
;	DX					(max X)
; Out:	none
; Use:	none
; Modf:	BX
; Call:	@setnewrange
;
hrange_07	proc
		mov	bx,offset X
		j	@setnewrange
hrange_07	endp

;
; 08 - Set vertical cursor range
;
;
; In:	CX					(min Y)
;	DX					(max Y)
; Out:	none
; Use:	pos
; Modf:	CX, DX, BX, rangemin, rangemax
; Call:	setpos_04
;
vrange_08	proc
		mov	bx,offset Y
@setnewrange:	cmp	cx,dx
		jle	@@saverange
		xchg	cx,dx
@@saverange:	mov	word ptr rangemin[bx],cx
		mov	word ptr rangemax[bx],dx
		mov	cx,[pos.X]
		mov	dx,[pos.Y]
		;j	setpos_04
vrange_08	endp

;
; 04 - Position mouse cursor
;
;
; In:	CX					(X - column)
;	DX					(Y - row)
; Out:	none
; Use:	cursorhidden, granumask.Y
; Modf:	AX, CX, SI, videounlock, cursordrawn
; Call:	@setpos
;
setpos_04	proc
		mov	si,LenZeroArea1/2	; clear area 1
ERRIF (LenZeroArea1 mod 2 ne 0) "LenZeroArea1 must be even!"
		call	@setpos

showcursor:	cmp	[cursorhidden],0
@showcheck:	jnz	@@showret		; exit if cursor hidden

@showcursor:	mov	dx,offset drawcursor
@cursorproc:	dec	[videounlock]
		jnz	@@showdone		; exit if drawing in progress
		xor	cx,cx
		xchg	cl,[cursordrawn]	; mark cursor as not drawn
		mov	ax,[granumask.Y]
		inc	ax			; Zero flag if graphics
		call	dx			; CX=old cursordrawn

@@showdone:	inc	[videounlock]		; drawing stopped
@@showret:	ret
setpos_04	endp

;
; 09 - Define graphics cursor
;
;
; In:	BX					(hot spot X)
;	CX					(hot spot Y)
;	ES:DX					(pointer to bitmaps)
; Out:	none
; Use:	none
; Modf:	AX, CX, BX, SI, DI, ES, hotspot, screenmask, cursormask
; Call:	@retcursor, hidecursor, showcursor
;
graphcursor_09	proc
;---------- compare user shape with internal area
		mov	si,offset hotspot
		lodsw
		cmp	ax,bx
		jne	@@redrawgraph
		lodsw
		xor	ax,cx
		jne	@@redrawgraph
		mov	di,dx
		mov	al,16+16		; OPTIMIZE: AL instead AX
		xchg	ax,cx
		repe	cmpsw
		xchg	cx,ax			; OPTIMIZE: instead MOV CX,AX
		je	@retcursor

;---------- copy user shape to internal area
@@redrawgraph:	push	ds ds es
		pop	ds es
		mov	di,offset hotspot
		xchg	ax,bx			; OPTIMIZE: instead MOV AX,BX
		stosw
		xchg	ax,cx			; OPTIMIZE: instead MOV AX,CX
		stosw
		mov	si,dx
		mov	cx,16+16
		rep	movsw
		pop	ds

redrawscreen:	call	hidecursor
		j	showcursor
graphcursor_09	endp

;
; 0A - Define text cursor
;
;
; In:	BX					(0 - SW, else HW text cursor)
;	CX					(screen mask/start scanline)
;	DX					(cursor mask/end scanline)
; Out:	none
; Use:	none
; Modf:	AX, CX, BX, cursortype, startscan, endscan
; Call:	INT 10/01, @retcursor, redrawscreen
;
textcursor_0A	proc
		or	bl,bh
		jz	@@difftext		; jump if software cursor

		push	cx
		mov	ch,cl
		mov	cl,dl
		mov	ah,1
		int	10h			; set cursor shape & size
		pop	cx
		mov	bl,1

@@difftext:	cmp	bl,[cursortype]
		jne	@@redrawtext
		cmp	cx,[startscan]
		jne	@@redrawtext
		cmp	dx,[endscan]
		je	@retcursor		; exit if cursor not changed

@@redrawtext:	mov	[cursortype],bl
		mov	[startscan],cx
		mov	[endscan],dx
		j	redrawscreen
textcursor_0A	endp

;
; 0D - Light pen emulation On
;
;
; In:	none
; Out:	none
; Use:	none
; Modf:	none
; Call:	lightpenoff_0E
;
;;+*lightpenon_0D	proc
;;+*		mov	al,0
;;+*		;j	lightpenoff_0E
;;+*lightpenon_0D	endp

;
; 0E - Light pen emulation Off
;
;
; In:	none
; Out:	none
; Use:	none
; Modf:	nolightpen?
; Call:	none
;
;;+*lightpenoff_0E	proc
;;+*		mov	[nolightpen?],al	; OPTIMIZE: AL instead nonzero
;;+*		ret
;;+*lightpenoff_0E	endp

;
; 0F - Set mickeys/pixels ratios
;
;
; In:	CX					(number of mickeys per 8 pix
;	DX					 horizontally/vertically)
; Out:	none
; Use:	none
; Modf:	mickey8
; Call:	none
;
micperpixel_0F	proc
		mov	[mickey8.X],cx
		mov	[mickey8.Y],dx
@retcursor:	ret
micperpixel_0F	endp

;
; 10 - Define screen region for updating
;
;
; In:	CX, DX					(X/Y of upper left corner)
;	SI, DI					(X/Y of lower right corner)
; Out:	none
; Use:	none
; Modf:	CX, DX, SI, DI, upleft, lowright
; Call:	redrawscreen
;
defregion_10	proc
		mov	ax,[_ARG_SI_]
		cmp	cx,ax
		jle	@@regionX
		xchg	cx,ax
@@regionX:	mov	[upleft.X],cx
		mov	[lowright.X],ax
		xchg	ax,di			; OPTIMIZE: instead MOV AX,DI
		cmp	dx,ax
		jle	@@regionY
		xchg	dx,ax
@@regionY:	mov	[upleft.Y],dx
		mov	[lowright.Y],ax
		j	redrawscreen
defregion_10	endp

;
; 14 - Exchange user interrupt routines
;
;
; In:	CX					(new call mask)
;	ES:DX					(new FAR routine)
; Out:	[CX]					(old call mask)
;	[ES:DX]					(old FAR routine)
; Use:	callmask, userproc
; Modf:	AX
; Call:	intpar_0C
;
exchangeint_14	proc
		mov	al,[callmask]
		;mov	ah,0
		mov	[_ARG_CX_],ax
		mov	ax,word ptr userproc[0]
		mov	[_ARG_DX_],ax
		mov	ax,word ptr userproc[2]
		mov	[_ARG_ES_],ax
		;j	intpar_0C
exchangeint_14	endp

;
; 0C - Define user interrupt routine
;
;
; In:	CX					(call mask)
;	ES:DX					(FAR routine)
; Out:	none
; Use:	none
; Modf:	userproc, callmask
; Call:	none
;
intpar_0C	proc
		saveFAR [userproc],es,dx
		mov	[callmask],cl
		ret
intpar_0C	endp

;
; 16 - Save driver state
;
;
; In:	BX					(buffer size)
;	ES:DX					(buffer)
; Out:	none
; Use:	StartSaveArea
; Modf:	CX, SI, DI
; Call:	none
;
savestate_16	proc
		mov	di,dx
		mov	si,offset StartSaveArea
@saveareamove:	mov	cx,LenSaveArea
		cmp	bx,cx
		jb	@@saveret
		shr	cx,1
		rep	movsw
@@saveret:	ret
savestate_16	endp

;
; 17 - Restore driver state
;
;
; In:	BX					(buffer size)
;	ES:DX					(saved state buffer)
; Out:	none
; Use:	none
; Modf:	SI, DI, DS, ES, StartSaveArea
; Call:	@saveareamove
;
restorestate_17	proc
		push	ds es
		pop	ds es
		mov	si,dx
		mov	di,offset StartSaveArea
		j	@saveareamove
restorestate_17	endp

;
; 1A - Set mouse sensitivity
;
;
; In:	BX				(number of mickeys per 8 pix
;	CX				 horizontally/vertically)
;	DX				(speed threshold in mickeys/second)
; Out:	none
; Use:	none
; Modf:	mickey8
; Call:	doublespeed_13
;
setsens_1A	proc
		mov	[mickey8.X],bx
		mov	[mickey8.Y],cx
		;j	doublespeed_13
setsens_1A	endp

;
; 13 - Define double-speed threshold
;
;
; In:	DX				(speed threshold in mickeys/second)
; Out:	none
; Use:	none
; Modf:	/DX/, /doublespeed/
; Call:	none
;
doublespeed_13	proc
;;+*		or	dx,dx
;;+*		jnz	@@savespeed
;;+*		mov	dl,64
;;+*@@setspeed:	mov	[doublespeed],dx
		;ret
doublespeed_13	endp

;
; 11 12 18 19 1C - Null function for not implemented calls
;

nullfunc	proc
		ret
nullfunc	endp

;
;				INT 33 handler
;

		evendata
handler33table	dw	offset resetdriver_00
		dw	offset showcursor_01
		dw	offset hidecursor_02
		dw	offset status_03
		dw	offset setpos_04
		dw	offset butpresdata_05
		dw	offset buttreldata_06
		dw	offset hrange_07
		dw	offset vrange_08
		dw	offset graphcursor_09
		dw	offset textcursor_0A
		dw	offset getmcounter_0B
		dw	offset intpar_0C
		dw	offset nullfunc		;lightpenon_0D
		dw	offset nullfunc		;lightpenoff_0E
		dw	offset micperpixel_0F
		dw	offset defregion_10
		dw	offset nullfunc		;11 - genius driver only
		dw	offset nullfunc		;12 - large graphics cursor
		dw	offset doublespeed_13
		dw	offset exchangeint_14
		dw	offset storagereq_15
		dw	offset savestate_16
		dw	offset restorestate_17
		dw	offset nullfunc		;18 - set alternate handler
		dw	offset nullfunc		;19 - return alternate address
		dw	offset setsens_1A
		dw	offset getsens_1B
		dw	offset nullfunc		;1C - InPort mouse only
		dw	offset nullfunc		;1D - define display page #
		dw	offset getpage_1E
		dw	offset disabledrv_1F
		dw	offset enabledriver_20
		dw	offset softreset_21

handler33	proc	far
		sti
		cld
		or	ah,ah
		jnz	@@iret

;
; 23 - Get language for messages
; Out:	[BX]					(language code: 0 - English)
;
getlanguage_23:	cmp	al,23h
		je	@@iretBX0

;
; 32 - Get active advanced functions
; Out:	[AX]					(active functions flag)
;	[BX] = 0
;	[CX] = 0
;	[DX] = 0
;
getactive_32:	cmp	al,32h
		jne	no_getactive_32
		mov	ax,0110010000000100b	; active: 26 27 2A 32
		xor	cx,cx
		xor	dx,dx
@@iretBX0:	xor	bx,bx
@@iret:		iret
no_getactive_32:

		push	ds
		movSeg	ds,cs

;
; 24 - Get software version, mouse type and IRQ
; Out:	[AX] = 24h/FFFFh			(installed/error)
;	[BX]					(version)
;	[CH] = 1=bus/2=serial/3=InPort/4=PS2/5=HP (mouse type)
;	[CL]					(interrupt #/0=PS/2)
; Use:	driverversion
;
getversion_24:	cmp	al,24h
		jne	no_getversion_24
		mov	bx,driverversion
		db	0B9h			; MOV CX,word
mouseinfo	db	?,4
no_getversion_24:

;
; 26 - Get maximum virtual screen coordinates
; Out:	[BX]					(mouse disabled flag)
;	[CX]					(maximum virtual screen X)
;	[DX]					(maximum virtual screen Y)
; Use:	maxcoordY
; Call:	none
;
getmaxvirt_26:	cmp	al,26h
		jne	no_getmaxvirt_26
		db	0BBh			; MOV BX,word
disabled?	db	1,0			; 1 - driver disabled
		db	0B9h			; MOV CX,word
maxcoordX	dw	?			; screen width
		dec	cx
		mov	dx,[maxcoordY]
		dec	dx
no_getmaxvirt_26:

;
; 2A - Get cursor hot spot
; Out:	[AX]					(cursor visibility counter)
;	[BX]					(hot spot X)
;	[CX]					(hot spot Y)
;	[DX] = 1=bus/2=serial/3=InPort/4=PS2/5=HP (mouse type)
; Use:	cursorhidden, hotspot
;
gethotspot_2A:	cmp	al,2Ah
		jne	no_gethotspot_2A
		mov	al,[cursorhidden]
		mov	bx,[hotspot.X]
		mov	cx,[hotspot.Y]
		db	0BAh			; MOV DX,word
mouseinfo1	db	4,0
		j	@@iretDS
no_gethotspot_2A:

;
; 4D - Get pointer to copyright string
; Out:	[ES:DI]					(copyright string)
; Use:	IDstring
;
getcopyrght_4D:	cmp	al,4Dh
		jne	no_getcopyrght_4D
		movSeg	es,ds
		mov	di,offset IDstring
no_getcopyrght_4D:

;
; 6D - Get pointer to version
; Out:	[ES:DI]					(version string)
; Use:	msversion
;
getversion_6D:	cmp	al,6Dh
		jne	no_getversion_6D
		movSeg	es,ds
		mov	di,offset msversion
no_getversion_6D:

		cmp	al,0Bh
		je	getmcounter_0B

;
; 27 - Get screen/cursor masks and mickey counts
; Out:	[AX]			(screen mask/start scanline)
;	[BX]			(cursor mask/end scanline)
;	[CX]			(number of mickeys mouse moved
;	[DX]			 horizontally/vertically since last call)
; Call:	getmcounter_0B
;
getscreen_27:	cmp	al,27h
		jne	@@calltable33
		mov	ax,[startscan]
		mov	bx,[endscan]
		;j	getmcounter_0B

;
; 0B - Get motion counters
; Out:	[CX]			(number of mickeys mouse moved
;	[DX]			 horizontally/vertically since last call)
; Modf:	mickey
;
getmcounter_0B:	xor	cx,cx
		xchg	[mickey.X],cx
		xor	dx,dx
		xchg	[mickey.Y],dx
@@iretDS:	pop	ds
		iret
;----------
@@calltable33:	cmp	al,21h
		ja	@@iretDS
		push	ax cx dx bx bp si di es
		mov	bp,sp
		mov	si,ax			;!!! AX must be unchanged
		shl	si,1
		call	handler33table[si]	; call by calculated offset
@rethandler:	pop	es di si bp bx dx cx ax ds
		iret
handler33	endp

; END OF INT 33 SERVICES 


RILversion	label
msversion	db	driverversion / 100h,driverversion MOD 100h
IDstring	db	'CuteMouse 1.8',0
IDstringlen = $ - IDstring

EndTSRpart	label


; INITIALIZATION PART DATA 

.data

paragraphs	dw	0
SaveMemStrat	dw	0
SaveUMBLink	dw	0
XMSentry	dd	0

mousetype	db	?			; 0=PS/2,1=MSys,2=LT,3=MS,4=WM
options		db	0
OPT_PS2		equ	00000001b
OPT_SERIAL	equ	00000010b
OPT_COMforced	equ	00000100b
OPT_PS2after	equ	00001000b
OPT_3button	equ	00010000b
OPT_NOMSYS	equ	00100000b
OPT_LEFTHAND	equ	01000000b
OPT_NOUMB	equ	10000000b

OPTION		struc
  C		db	?
  M		db	0
  PROC@		dw	?
	ends

		evendata
OPTABLE		OPTION	{C='P', M=OPT_PS2,	PROC@=@newopt}
		OPTION	{C='S', M=OPT_SERIAL,	PROC@=_serialopt}
;;+		OPTION	{C='A', M=OPT_SERIAL,	PROC@=_addropt}
		OPTION	{C='Y', M=OPT_NOMSYS,	PROC@=@newopt}
		OPTION	{C='V', M=OPT_PS2after,	PROC@=@newopt}
		OPTION	{C='3' and not 20h,M=OPT_3button,PROC@=@newopt}
		OPTION	{C='R',			PROC@=_resolution}
		OPTION	{C='L', M=OPT_LEFTHAND,	PROC@=@newopt}
		OPTION	{C='B',			PROC@=_checkdriver}
		OPTION	{C='W', M=OPT_NOUMB,	PROC@=@newopt}
		OPTION	{C='U',			PROC@=unloadTSR}
		OPTION	{C='?' and not 20h,	PROC@=EXITMSG}
OPTABLEcnt	equ	($-OPTABLE)/SIZE OPTION

INCLUDE ctmouse.msg

.code

; Real Start 

real_start:	cld
;---------- save old INT 33h handler
		mov	ax,3533h
		int	21h
		saveFAR [oldint33],es,bx

;---------- parse command line and find mouse
@@start:	PrintS	Copyright		; 'Cute Mouse Driver'
		call	commandline		; examine command line

		mov	cx,word ptr oldint33[2]
		jcxz	@@startfind
		mov	ax,1Fh
		int	33h			; disable old driver
;----------
@@startfind:	mov	al,[options]
		test	al,OPT_PS2+OPT_SERIAL
		jnz	@@find1COM
		or	al,OPT_PS2+OPT_SERIAL

@@find1COM:	test	al,OPT_PS2after
		jz	@@findPS2
		push	ax
		call	searchCOM
		pop	ax
		jnc	@@serialfound

@@findPS2:	test	al,OPT_PS2+OPT_PS2after
		jz	@@find2COM
		push	ax
		mov	al,0
		mov	mouseinfo[0],al
		mov	[mousetype],al
		call	checkPS2
		pop	ax
		jnc	@@mousefound

@@find2COM:	test	al,OPT_PS2after
		jnz	@@devnotfound
		test	al,OPT_SERIAL+OPT_NOMSYS
		jz	@@devnotfound
		call	searchCOM		; call if no /V and serial
		jnc	@@serialfound

@@devnotfound:	mov	dx,offset E_notfound	; 'Error: device not found'
		mov	cx,word ptr oldint33[2]
		jmp	EXITENABLE

@@serialfound:	mov	al,2
		mov	mouseinfo[1],al
		mov	[mouseinfo1],al

;---------- check if CuteMouse driver already installed
@@mousefound:	call	getCuteMouse
		mov	dx,offset S_reset	; 'Resident part reset to'
		mov	bl,2			; errorlevel shift
		je	@@setupdriver

;---------- allocate UMB memory, if possible, and set INT 33 handler
		call	prepareTSR		; new memory segment in ES
		mov	si,offset oldint33	; copy oldint33 contents
		mov	di,si
		movsw
		movsw
		push	ds
		movSeg	ds,es
		mov	[disabled?],1
		mov	ax,2533h
		mov	dx,offset handler33
		int	21h			; setup new INT 33 handler
		pop	ds
		mov	dx,offset S_installed	; 'Installed at'
		mov	bl,0			; errorlevel shift
;----------
@@setupdriver:	PrintS
		mov	cl,[mousetype]
		mov	dx,offset S_forPS2	; 'PS/2 port'
		or	cl,cl
		jz	@@printmode

		PrintS	S_atCOM
		mov	dx,offset S_forWM	; 'Microsoft wheel mouse'
		cmp	cl,4
		je	@@printmode
		mov	dx,offset S_forMS	; 'Microsoft mouse'
		cmp	cl,2
		ja	@@printmode
		mov	dx,offset S_forLT	; 'Logitech MouseMan'
		je	@@printmode
		mov	dx,offset S_forMSYS	; 'Mouse Systems mouse'
		inc	bx			; OPTIMIZE: BX instead BL

@@printmode:	mov	exitfunc[0],bl
		PrintS
		call	setupdriver
		xor	ax,ax
		int	33h			; Reset driver

		mov	ax,es
		mov	bx,cs
		cmp	ax,bx
		jne	@@exit			; jump if TSR copied to UMB

;---------- TSR exit if no other memory grabbed
		mov	es,ds:[2Ch]		; release environment
		mov	ah,49h
		int	21h

		mov	dx,offset EndTSRpart+15
		mov	cl,4
		shr	dx,cl
		mov	exitfunc[1],31h		; TSR exit, al=return code
@@exit:		jmp	EXIT

;
;		Setup resident driver code and parameters
;

setupdriver	proc
		;mov	cl,[mousetype]

;---------- detect VGA card (VGA ATC have one more register)
		mov	ax,1A00h
		int	10h			; get display type in bx
		cmp	al,1Ah
		jne	@@lhand
		xchg	ax,bx			; OPTIMIZE: instead MOV AL,BL
		cmp	al,7			; monochrome VGA?
		je	@@setATCVGA
		cmp	al,8			; color VGA?
		jne	@@lhand
@@setATCVGA:	inc	videoregs@[(SIZE RGROUPDEF)*3].regscnt

;---------- setup left hand mode handling
@@lhand:	mov	al,0EBh			; JMP SHORT
		mov	bl,07Ah			; JPE
		or	cl,cl
		jz	@@checklhand		; jump if PS/2 mouse
		xchg	ax,bx
@@checklhand:	test	[options],OPT_LEFTHAND
		jz	@@setlhand
		xchg	ax,bx
@@setlhand:	mov	[LEFTHANDCODE],al

;---------- setup buttons count and mask
		mov	al,3
		test	[options],OPT_3button
		jnz	@@sethandler
		or	cl,cl
		je	@@setbuttons		; jump if PS/2 mouse
		cmp	cl,3
		jne	@@sethandler		; jump if no MS mode
@@setbuttons:	mov	[buttonsmask],al	; OPTIMIZE: AL instead 3
		dec	ax			; OPTIMIZE: instead MOV AL,2
		mov	[buttonscnt],al		; OPTIMIZE: AL instead 2

;---------- setup mouse handlers code
@@sethandler:	cmp	cl,1
		jb	@@setother		; jump if PS/2 mouse

		mov	word ptr [IRQproc],20B0h ; MOV AL,20h
		mov	[BUFFADDR],offset bufferSERIAL
		mov	[enableproc],enableCOM-enableproc-2
		mov	[disableproc],disableCOM-disableproc-2
		je	@@setother		; jump if Mouse Systems mouse
		mov	[mouseproc],MSLTproc-mouseproc-2
		cmp	cl,3
		jb	@@setother		; jump if Logitech mouse
		mov	MSLTCODE3[1],2
		ja	@@setother		; jump if Intellimouse

		cmp	al,2
		je	@@setMSproc		; jump if MS2
		mov	[MSLTCODE2],075h	; JNZ
@@setMSproc:	mov	al,0C3h			; RET
		mov	[MSLTCODE1],al
		mov	[MSLTCODE3],al

;---------- setup, if required, other parameters
@@setother:	push	es ds es ds
		pop	es ds
		mov	si,offset oldint10	; save back [oldint10]...
		mov	di,si
		movsw
		movsw
		mov	al,[disabled?]		; ...and [disabled?]
		pop	ds
		mov	[disabled?],al

		mov	al,[IRQintnum]		; save old IRQ handler
		mov	ah,35h
		int	21h			; get INT in ES:BX
		mov	ax,es
		pop	es
		mov	di,offset oldIRQaddr
		xchg	ax,bx
		stosw
		xchg	ax,bx			; OPTIMIZE: instead MOV AX,BX
		stosw

;---------- copy TSR image (even if in same place - this is not awful)
		mov	si,offset StartTSRpart	; relocate initialized
		mov	di,si			;  resident portion
		mov	cx,(EndTSRpart+1-StartTSRpart)/2
		rep	movsw
		ret
setupdriver	endp

;
;		Check given or all COM-ports for mouse connection
;

searchCOM	proc
		;mov	[COMLCR],2
		mov	di,offset detectmouse
		call	COMloop
		jnc	@searchret

		test	[options],OPT_NOMSYS
		stc
		jnz	@searchret

		mov	[COMLCR],3
		mov	[mousetype],1		; set Mouse Systems mode
		mov	di,offset COMexist?
		;j	COMloop
searchCOM	endp

;

COMloop		proc
		mov	cx,1			; scan only current COM port
		test	[options],OPT_COMforced
		jnz	@@checkCOM
		mov	cl,4			; scan all COM ports

@@COMloop:	push	cx
		mov	al,'5'
		sub	al,cl
		call	setCOMport
		pop	cx
@@checkCOM:	push	cx di
		mov	si,[IO_address]
		call	di
		pop	di cx
		jnc	@searchret
		loop	@@COMloop
@searchret:	ret
COMloop		endp

;
;			Check if COM port available
;
;
; In:	SI = [IO_address]
; Out:	Carry flag				(COM port not exist)
; Use:	none
; Modf:	AX, DX
; Call:	none
;
COMexist?	proc
		or	si,si
		stc
		jz	@searchret
		mov	dx,si
		add	dx,3
		mov	al,0
		cli
		out	dx,al			; {3FBh} reset comm params
		inc	dx
		in	al,dx			; {3FCh} modem ctrl reg
		mov	ah,al
		sub	dx,3
		in	al,dx			; {3F9h} int enable reg
		sti
		and	ax,0E0F0h		; get reserved bits (zero, if
		neg	ax			; port exists); nonzero makes
		ret				; carry flag
COMexist?	endp

;
;			Check if mouse available
;
;
; In:	SI = [IO_address]
; Out:	Carry flag				(no COM or mouse found)
; Use:	COMLCR
; Modf:	AX, CX, DX, BX, mousetype
; Call:	COMexist?, pulseRTS, readCOMbyte
;
detectmouse	proc
		call	COMexist?
		jc	@searchret

		mov	dx,si
		add	dx,3
		mov	al,80h
		out	dx,al			; {3FBh} set DLAB on

		xchg	dx,si
		mov	ax,96
		out	dx,ax			; {3F8h},{3F9h} 1200 baud

		xchg	dx,si
		mov	al,[COMLCR]
		out	dx,al			; {3FBh} DLAB off, no parity,
						;  stop=1, length=7/8
		dec	dx
		dec	dx
		mov	al,0
		out	dx,al			; {3F9h} all interrupts off
		add	dx,3			; {3FCh}
		call	pulseRTS		; drop and raise RTS signal

;---------- detect if Microsoft or Logitech mouse present
		mov	bx,301h			; bl=no MS, bh=mousetype
		mov	cx,4			; scan 4 first bytes
		inc	dx			; DX=3FDh, SI=3F8h
@@detectloop:	call	readCOMbyte
		jc	@@detdone
		cmp	al,8
		je	@@skipdata		; jump if PNP data starts
		cmp	al,'M'
		jne	@@checkWM
		mov	bl,0			; MS compatible mouse found...
@@checkWM:	cmp	al,'Z'
		jne	@@checkLT
		mov	bh,4			; ...Intellimouse mouse found
@@checkLT:	cmp	al,'3'
		jne	@@detectnext
		mov	bh,2			; ...Logitech mouse found
@@detectnext:	loop	@@detectloop

@@skipdata:	call	readCOMbyte
		jnc	@@skipdata

@@detdone:	mov	[mousetype],bh		; save 2=LT, 3=MS or 4=WM
		neg	bl			; nonzero makes carry flag
		ret
detectmouse	endp

;
; In:	DX = [IO_address]+4
; Out:	ES = 0
; Modf:	AX, DI
;
pulseRTS	proc
		xor	ax,ax
		mov	es,ax
		call	@@tickOUT		; DTR/RTS/OUT2 off
		mov	al,3			; DTR/RTS on, OUT2 off

@@tickOUT:	mov	di,es:[46Ch]
@@tickwait:	cmp	di,es:[46Ch]
		je	@@tickwait		; wait timer tick change
		out	dx,al			; {3FCh} change DTR/RTS/OUT2
		ret
pulseRTS	endp

;
; In:	DX = [IO_address]+5
;	SI = [IO_address]
;	ES = 0
; Out:	Carry flag				(timeout happens)
;	AL					(data, if no timeout)
; Modf:	AH, DI
;
readCOMbyte	proc
		mov	ah,2+1			; length of silence in ticks
						; (include rest of curr tick)
@@timeloop:	mov	di,es:[46Ch]
@@waitloop:	in	al,dx			; {3FDh} line status reg
		test	al,1
		jnz	@@readret		; jump if data ready
		cmp	di,es:[46Ch]
		je	@@waitloop		; jump if same tick
		dec	ah
		jnz	@@timeloop		; wait next tick of silence
@retC:		stc
		ret
@@readret:	xchg	dx,si
		in	al,dx			; {3F8h} receive byte
		xchg	dx,si
		;clc
		ret
readCOMbyte	endp

;
;				Set Mouse Port
;
;
; In:	AL					(COM port letter)
; Out:	none
; Use:	0:400
; Modf:	AL, BX, ES, com_port, IO_address
; Call:	setIRQ
;
setCOMport	proc
		mov	[com_port],al
		sub	al,'1'

		xor	bx,bx
		mov	es,bx
		mov	bl,al
		shl	bx,1
		mov	bx,es:400h[bx]
		mov	[IO_address],bx

		shr	al,1
		mov	al,4			; IRQ4 for COM1/3
		jnc	setIRQ
		dec	ax			; OPTIMIZE: instead MOV AL,3
		;j	setIRQ			; IRQ3 for COM2/4
setCOMport	endp

;
;				Set IRQ number
;
;
; In:	AL					(IRQ number)
; Out:	none
; Use:	none
; Modf:	AL, CL, mouseinfo, IRQintnum, IRQintnum2, PICstate, notPICstate
; Call:	none
;
setIRQ		proc
		mov	mouseinfo[0],al
		mov	cl,al
		add	al,8
		mov	[IRQintnum],al
		mov	[IRQintnum2],al
		mov	al,1
		shl	al,cl
		mov	[PICstate],al		; PIC interrupt disabler
		not	al
		mov	[notPICstate],al	; PIC interrupt enabler
		ret
setIRQ		endp

;
;				Check for PS/2
;
;
; In:	none
; Out:	Carry flag
; Use:	none
; Modf:	AX, CX, BX
; Call:	none
;
checkPS2	proc
		mov	bh,3
		PS2serv 0C205h,@retC		; initialize mouse, bh=datasize
		mov	bh,3
		PS2serv 0C203h,@retC		; set mouse resolution bh
		movSeg	es,cs
		mov	bx,offset IRQhandler
		PS2serv	0C207h,@retC		; mouse, es:bx=ptr to handler
		xor	bx,bx
		mov	es,bx
		PS2serv	0C207h			; mouse, es:bx=ptr to handler
		;clc
		ret
checkPS2	endp

;
;		Check if CuteMouse driver is installed
;
;
; In:	none
; Out:	Zero flag				(zero if installed)
;	ES					(driver segment)
; Use:	oldint33[2], IDstring
; Modf:	AX, CX, SI, DI
; Call:	INT 33/004D
;
getCuteMouse	proc
		mov	ax,word ptr oldint33[2]
		neg	ax
		sbb	ax,ax
		inc	ax
		jne	@ret			; jump if handler seg is zero
		mov	al,4Dh			; OPTIMIZE: AL instead AX
		int	33h
		cmp	di,offset IDstring
		jne	@ret
		mov	si,di
		mov	cx,IDstringlen
		rep	cmpsb
@ret:		ret
getCuteMouse	endp


; COMMAND LINE PARSING 

;
;			Parse Serial option
;

_serialopt	proc
		cmp	si,di
		jae	@ret
		lodsb
		cmp	al,'1'
		jb	@newopt2
		cmp	al,'4'
		ja	@newopt2
		or	[options],OPT_COMforced
		call	setCOMport		; '/Sc' -> set COM port
		cmp	si,di
		jae	@ret
		lodsb
		cmp	al,'2'
		jb	@newopt2
		cmp	al,'7'
		ja	@newopt2
		sub	al,'0'
		call	setIRQ			; '/Sci' -> set IRQ line
		j	@newopt
_serialopt	endp

;
;			Parse Resolution option
;

_resolution	proc
		;mov	ah,0
		cmp	si,di
		jae	@@dupres
		lodsb				; first argument
		sub	al,'0'
		cmp	al,9
		ja	@@fixupres		; jump if AL < 0 || AL > 9

		mov	ah,al
		cmp	si,di
		jae	@@setres
		lodsb				; second argument
		sub	al,'0'
		cmp	al,9
		jbe	@@setres		; jump if AL >= 0 && AL <= 9

@@fixupres:	dec	si			; fixup for command line ptr
@@dupres:	mov	al,ah			; drop invalid argument
@@setres:	mov	[mresolution0],ah
		mov	[mresolution1],al
		j	@newopt
_resolution	endp

;
;		Check if mouse services already present
;

_checkdriver	proc
		mov	cx,word ptr oldint33[2]
		jcxz	@newopt
		mov	ax,21h
		int	33h
		inc	ax
		jnz	@newopt
		mov	dx,offset E_mousepresent ; 'Mouse service already...'
		j	EXITMSG
_checkdriver	endp

;

commandline	proc
		mov	si,80h
		lodsb
		cbw				; OPTIMIZE: instead MOV AH,0
		mov	di,si
		add	di,ax

@newopt:	cmp	si,di
		jae	@ret
		lodsb
@newopt2:	cmp	al,' '
		jbe	@newopt
		cmp	al,'/'			; option character?
		jne	@@badoption
		cmp	si,di			; option itself there?
		jae	@@badoption
		lodsb
		and	al,not 20h

		mov	dx,offset Syntax	; 'Options:'
		mov	bx,offset OPTABLE
		mov	cx,OPTABLEcnt
@@optloop:	cmp	al,[bx].C
		jne	@@nextopt
		mov	ah,[bx].M
		or	[options],ah
		j	[bx].PROC@
@@nextopt:	add	bx,SIZE OPTION
		loop	@@optloop

@@badoption:	mov	dx,offset E_option	; 'Error: Invalid option'

EXITMSG:	mov	bx,dx
		mov	al,[bx]
		mov	exitfunc[0],al
		inc	dx
		PrintS
EXIT:		db	0B8h			; MOV AX,word
exitfunc	db	?,4Ch			; terminate, al=return code
		int	21h
commandline	endp


; TSR MANAGEMENT 

;
;			Unload driver and quit
;

unloadTSR	proc
		call	getCuteMouse		; check if CTMOUSE installed
		mov	dx,offset E_nocute	; 'CuteMouse driver is not installed!'
		jne	EXITMSG

		push	es
		mov	ax,1Fh
		int	33h			; disable CuteMouse driver
		mov	cx,es
		pop	es
		cmp	al,1Fh
		mov	dx,offset E_notunload	; 'Driver unload failed...'
		jne	@@exitenable

		push	ds
		mov	ds,cx
		mov	dx,bx
		mov	ax,2533h
		int	21h			; restore old int33 handler
		pop	ds
		call	FreeMem
		mov	dx,offset S_unloaded	; 'Driver successfully unloaded...'
EXITENABLE:	jcxz	EXITMSG			; check if old driver exist
@@exitenable:	mov	ax,20h
		int	33h			; enable old driver
		j	EXITMSG
unloadTSR	endp

;

prepareTSR	proc
		mov	ax,offset EndTSRpart	; get number of bytes
		call	AllocUMB		;  we need memory for
		mov	ax,es
		mov	bx,cs
		cmp	ax,bx
		je	@@prepareret		; exit if TSR "in place"

		push	ds
		dec	ax
		mov	es,ax			; target MCB segment...
		inc	ax
		mov	es:[1],ax		; ...modify owner field
		dec	bx
		mov	ds,bx			; current MCB
		mov	si,8
		mov	di,si
		movsw				; copy process name
		movsw
		movsw
		movsw
		pop	ds

		mov	bx,ax
		add	bx,es:[3]		; add target MCB size field
		mov	es,ax			; target PSP
		xchg	dx,ax			; OPTIMIZE: instead MOV DX,AX
		mov	ah,26h
		int	21h			; create PSP
		mov	es:[2],bx		; fix upper segment number
@@prepareret:	ret
prepareTSR	endp


; MEMORY HANDLING 

;
; Get "XMS" handler address (2 words)
;
; In:	none
; Out:	Zero flag
; Use:	none
; Modf:	AX, BX, XMSentry
; Call:	INT 2F/4310
;
getXMSaddr	proc
		push	es
		xor	bx,bx
		mov	es,bx
		mov	ax,4310h		; XMS: Get Driver Address
		int	2Fh
		mov	ax,es
		saveFAR [XMSentry],ax,bx
		or	ax,bx			; ZF indicates error: JZ error
		pop	es
		ret
getXMSaddr	endp

;
; Save Allocation Srategy
;
; In:	none
; Out:	Carry flag				(no UMB link supported)
; Use:	none
; Modf:	AX, SaveMemStrat, SaveUMBLink
; Call:	INT 21/5800, INT 21/5802
;
SaveStrategy	proc
		mov	ax,5800h
		int	21h			; get DOS memory alloc strategy
		mov	[SaveMemStrat],ax
		mov	ax,5802h
		int	21h			; get UMB link state
		mov	byte ptr [SaveUMBLink],al
		ret
SaveStrategy	endp

;
; Restore allocation strategy
;
; In:	none
; Out:	none
; Use:	SaveMemStrat, SaveUMBLink
; Modf:	AX, BX
; Call:	INT 21/5801, INT 21/5803
;
RestoreStrategy	proc
		mov	bx,[SaveMemStrat]
		mov	ax,5801h
		int	21h			; set DOS memory alloc strategy
		mov	bx,[SaveUMBLink]
		mov	ax,5803h
		int	21h			; set UMB link state
		ret
RestoreStrategy	endp

;
; Allocate high memory
;
; In:	AX					(memory size required)
; Out:	ES					(grabbed memory segment)
; Use:	XMSentry
; Modf:	AX, CX, DX, BX, paragraphs
; Call:	INT 21/30, INT 21/48, INT 21/49, INT 21/5801, INT 21/5803,
;	INT 2F/4300, SaveStrategy, RestoreStrategy, getXMSaddr
;
AllocUMB	proc
		add	ax,15
		mov	cl,4
		shr	ax,cl
		mov	[paragraphs],ax

		test	[options],OPT_NOUMB
		jnz	@@allocasis		; jump if UMB prohibited
		mov	ax,cs
		cmp	ah,0A0h
		jae	@@allocasis		; jump if already loaded hi

;---------- check if UMB is DOS type
		call	SaveStrategy
		mov	bx,1			; add UMB to MCB chain
		mov	ax,5803h
		int	21h			; set UMB link state

; try to set a best strategy to allocate DOS supported UMBs
		mov	bl,41h			; OPTIMIZE: BL instead BX
		mov	ax,5801h		; hi mem, best fit
		int	21h			; set alloc strategy
		jnc	@@allocDOSUMB

; try a worse one then
		mov	bl,81h			; OPTIMIZE: BL instead BX
		mov	ax,5801h		; hi mem then low mem, best fit
		int	21h			; set alloc strategy

@@allocDOSUMB:	mov	bx,[paragraphs]
		mov	ah,48h
		int	21h			; allocate UMB memory
		pushf
		push	ax
		call	RestoreStrategy		; restore allocation strategy
		pop	ax
		popf
		jc	@@allocXMSUMB
		mov	es,ax
		cmp	ah,0A0h			; jump if allocated mem is
		jae	@@allocret		;  is above 640k,
		mov	ah,49h			;  else free it
		int	21h

;---------- try XMS driver to allocate UMB
@@allocXMSUMB:	call	getXMSaddr
		jz	@@allocasis
		mov	ah,10h			; XMS: Request UMB
		mov	dx,[paragraphs]
		call	[XMSentry]
		or	ax,ax
		jz	@@allocasis
		mov	es,bx			; memory segment
		cmp	dx,[paragraphs]
		jae	@@allocret
		mov	dx,es
		mov	ah,11h			; XMS: Release UMB
		call	[XMSentry]

;---------- use current memory segment
@@allocasis:	movSeg	es,cs

@@allocret:	ret
AllocUMB	endp

;
; In:	ES					(segment to free)
; Out:	none
; Use:	XMSentry
; Modf:	AH, DX
; Call:	INT 21/49, getXMSaddr
;
FreeMem		proc
		call	getXMSaddr
		jz	@@freeDOS

		mov	dx,es
		mov	ah,11h			; XMS: Release UMB
		call	[XMSentry]

@@freeDOS:	mov	ah,49h
		int	21h			; free allocated memory
		ret
FreeMem		endp

;

		end	start
